// $Id: TransferMatrix.h 1252 2021-12-20 14:43:47Z ge $
/// \file TransferMatrix.h
/// \brief contains the Hamiltonian function class
///
/// $Revision: 1252 $
/// \author Gerald Weber <gweberbh@gmail.com>
#ifndef GBC_TRANSFERMATRIX3D_H
#define GBC_TRANSFERMATRIX3D_H "$Id: TransferMatrix.h 1252 2021-12-20 14:43:47Z ge $"
#include <valarray>
#include <map>
#include "Range.h"
#include "Hamiltonian.h"
#include "TestModel.h"
#include "Quadrature.h"
#include "EigenSystem.h"
#include <gsl/gsl_const_mksa.h>
#include <boost/filesystem.hpp>
#include "ErrorCodes.h"


namespace gbc
  {
  template<class _Tp=double>
  /// \brief Calculation of Hamiltonians using the Transfer Matrix technique.
  ///
  /// Add new hamiltonians as objects (void constructor), add an short identification
  /// to hamiltonian_id and ammend the method select_hamiltonian and K.
  /// Also provide a specific parameter method for the new Hamiltonian.
  class TransferMatrix
    {
    public:
      typedef _Tp                              value_type;
      typedef std::valarray<value_type>        vector_type;
      typedef valmatrix<value_type>            matrix_type;
      typedef std::map<value_type,vector_type>  result_map_type;
      typedef Quadrature<Legendre<std::valarray<value_type> > > quadrature_type;

//    enum hamiltonian_id {pb,dpb,hms,jb,mes,test,zzzz}; Add new identification zzzz here, lowercase
      enum hamiltonian_id {pb,dpb,hms,jb,mes,test,pb3DA,pcla,pclj,trmf};  ///< The short identification for Hamiltonians. *** Add new identification here, lowercase
      typedef std::map<std::string,hamiltonian_id> hamiltonian_map_type;

      bool Debug;                           ///< Flag for debugging
      bool Periodic_boundary_condition;     ///< Flag for boundary conditions
      bool Recalculate_diagonalization;     ///< If true recalculates the diagonalization
      bool Parameters_changed;              ///< Flags if parameters changed
      size_t Eigenvalue_cutoff;             ///< Cut-off value for the number of Eigenvalues/vectors to be used in certain integration procedures.

    protected:
      value_type Beta;                      ///< The temperature divided by the Boltzman constant
      value_type Temperature;               ///< The temperature to be used for the hamiltonian
    public:
      static value_type Lower_limit;        ///< Lower integration limit
      static value_type Upper_limit;        ///< Upper integration limit
      static vector_type Integration_space; ///< Vector containg the points for integration, these are usually rescaled by the quadrature class
      static vector_type Weights;           ///< The integration calculated weights, depends on the selected quadrature.
      static vector_type SqrtWeights;       ///< The square root of Weights, needed for symmetrizing the matrix.
      static matrix_type W, YD, tmp_xy, Rsqrt, Rminus_sqrt, phi; ///< Working matrices
      size_t Integration_steps;      ///< The number of points used in the integration.
      quadrature_type* quadrature;          ///< The quadrature used for integration and formation of the eigensystem
      EigenSystem<value_type,value_type> eigen_engine; ///< The functions that diagonalizses the eigensystem
      
      // Hamiltoninans
      hamiltonian_id Hamiltonian_ID;        ///< The identification of the Hamiltonian to be used/
      hamiltonian_map_type Available_hamiltonians; ///< A map relating names and hamiltonian_id
      PeyrardBishop<value_type>        PB;
      DauxoisPeyrardBishop<value_type> DPB;
      HarmonicMorseSolvent<value_type> HMS;
      JoyeuxBuyukdagli<value_type>     JB;
      MorseExactStacking<value_type>   MES; 
      TestHamiltonian<value_type>   TEST;
      PeyrardBishop3DA<value_type>        PB3DA;
      PeyrardCuestaLopezAngelov<value_type> PCLA;
      PeyrardCuestaLopezJames<value_type> PCLJ;
      TapiaRojoMazoFalo<value_type>    TRMF;
//    Add new identification here, uppercase,XXXX=the name of the new class in Hamiltonian.h,ZZZZ=identification here, same as zzzz but uppercase
//    XXXX<value_type>   ZZZZ; 

      value_type Position;
      std::string      ParameterIdName;
      std::string      MatrixDir;


      TransferMatrix(void)
       : Debug(false), Periodic_boundary_condition(true), Parameters_changed(true), Recalculate_diagonalization(true),
         Eigenvalue_cutoff(), Temperature(), Integration_steps(),
         Hamiltonian_ID(pb)
       {
       set_available_hamiltonians();
       eigen_engine.Decrease_sort_eigenvalues=true;
       quadrature = new quadrature_type();
       quadrature->Rule.Save_calculations=true;
       quadrature->Rule.Load_calculations=true;
       }

      TransferMatrix(const BasePairNeighbours<> &bpn)
       : Debug(false), Periodic_boundary_condition(true), Parameters_changed(true), Recalculate_diagonalization(true),
         Eigenvalue_cutoff(), Temperature(), Integration_steps(0),
         Hamiltonian_ID(pb), PB(bpn), DPB(bpn), HMS(bpn), JB(bpn), MES(bpn), TEST(bpn), PB3DA(bpn), PCLA(bpn), PCLJ(bpn), TRMF(bpn) // *** Add new Hamiltonian here, eg NEWHAM(bpn)
       {
       set_available_hamiltonians();
       eigen_engine.Decrease_sort_eigenvalues=true;
       quadrature = new quadrature_type();
       quadrature->Rule.Save_calculations=true;
       quadrature->Rule.Load_calculations=true;
       }
       
    void set_available_hamiltonians(void)
      {
      // *** Add new Hamiltonian to this list
       // available_hamiltonians = { {"pb",pb}, {"dpb",dpb}, {"hms",hms}, {"jb",jb}, {"mes",mes}, {"test",test}, {"zzzz",zzzz}};
      Available_hamiltonians = { {"pb",pb}, {"dpb",dpb}, {"hms",hms}, {"jb",jb}, {"mes",mes}, {"test",test}, {"pb3DA",pb3DA}, {"pcla",pcla}, {"pclj",pclj}, {"trmf",trmf} };
      }

    void eigenvalue_cutoff(size_t c) {Eigenvalue_cutoff=c;}

    /// \brief Selects the Hamiltonian to be used in the calculation.
    ///
    /// \attention This method needs to be modified if new Hamiltonians are added.
    void select_hamiltonian(std::string hamiltonian_name)
      {
      if (Available_hamiltonians.find(hamiltonian_name) == Available_hamiltonians.end())
        {
        CERR_ERROR(ERRMHNDNE) " Model " << hamiltonian_name << " does not exists" << std::endl;
        CERR << "Available models are: "; 
        typename std::map<std::string,hamiltonian_id>::const_iterator it;
        for(auto& it : Available_hamiltonians) CERR << it.first << " ";
        CERR << std::endl;
        CERR_TERM
        }
      else
        {
        hamiltonian_id new_id=Available_hamiltonians[hamiltonian_name];
        if (new_id != Hamiltonian_ID) 
          {
          Recalculate_diagonalization=true;
          Hamiltonian_ID=new_id;
          }
        }
      }

    inline std::string hamiltonian(void)
      {
      switch(Hamiltonian_ID)
        {
        case pb   :  return std::string("pb");
        case dpb  :  return std::string("dpb");
        case hms  :  return std::string("hms");
        case jb   :  return std::string("jb");
        case mes  :  return std::string("mes");
        case test :  return std::string("test");
        case pb3DA :  return std::string("pb3DA");
        case pcla :  return std::string("pcla");
        case pclj :  return std::string("pclj");
        case trmf :  return std::string("trmf");
        // *** Add new Hamiltonian here
        // e.g.: case zzzz :  return std::string("zzzz");
        }

      return std::string();
      }

    inline bool get_hamiltonian_parameters(ParameterMap<value_type>& pm)
      {
      ParameterIdName=pm.Identify;
      switch(Hamiltonian_ID)
        {
        case pb   :  Parameters_changed = PB.get_potential_parameters(pm);   break;
        case dpb  :  Parameters_changed = DPB.get_potential_parameters(pm);  break;
        case hms  :  Parameters_changed = HMS.get_potential_parameters(pm);  break;
        case jb   :  Parameters_changed = JB.get_potential_parameters(pm);   break;
        case mes  :  Parameters_changed = MES.get_potential_parameters(pm);  break;
        case test :  Parameters_changed = TEST.get_potential_parameters(pm); break;
        case pb3DA :  Parameters_changed = PB3DA.get_potential_parameters(pm); break;
        case pcla :  Parameters_changed = PCLA.get_potential_parameters(pm); break;
        case pclj :  Parameters_changed = PCLJ.get_potential_parameters(pm); break;
        case trmf :  Parameters_changed = TRMF.get_potential_parameters(pm); break;
        // *** Add new Hamiltonian here
        // e.g.: case zzzz :  return ZZZZ.get_potential_parameters(pm);
        }
      return Parameters_changed;
      }

    inline bool get_stacking_parameters(ParameterMap<value_type>& pm)
      {
      ParameterIdName=pm.Identify;
      switch(Hamiltonian_ID)
        {
        case pb   :  Parameters_changed = PB.stacking_interaction()->get_potential_parameters(pm);   break;
        case dpb  :  Parameters_changed = DPB.stacking_interaction()->get_potential_parameters(pm);  break;
        case hms  :  Parameters_changed = HMS.stacking_interaction()->get_potential_parameters(pm);  break;
        case jb   :  Parameters_changed = JB.stacking_interaction()->get_potential_parameters(pm);   break;
        case mes  :  Parameters_changed = MES.stacking_interaction()->get_potential_parameters(pm);  break;
        case test :  Parameters_changed = TEST.stacking_interaction()->get_potential_parameters(pm); break;
        case pb3DA :  Parameters_changed = PB3DA.stacking_interaction()->get_potential_parameters(pm); break;
        case pcla :  Parameters_changed = PCLA.stacking_interaction()->get_potential_parameters(pm); break;
        case pclj :  Parameters_changed = PCLJ.stacking_interaction()->get_potential_parameters(pm); break;
        case trmf :  Parameters_changed = TRMF.stacking_interaction()->get_potential_parameters(pm); break;
        // *** Add new Hamiltonian here
        // e.g.: case zzzz :  return ZZZZ.stacking_interaction()->get_potential_parameters(pm);
        }
      return Parameters_changed;
      }

    inline bool get_basepair_parameters(ParameterMap<value_type>& pm)
      {
      ParameterIdName=pm.Identify;
      switch(Hamiltonian_ID)
        {
        case pb   :  Parameters_changed = PB.basepair_interaction()->get_potential_parameters(pm);   break;
        case dpb  :  Parameters_changed = DPB.basepair_interaction()->get_potential_parameters(pm);  break;
        case hms  :  Parameters_changed = HMS.basepair_interaction()->get_potential_parameters(pm);  break;
        case jb   :  Parameters_changed = JB.basepair_interaction()->get_potential_parameters(pm);   break;
        case mes  :  Parameters_changed = MES.basepair_interaction()->get_potential_parameters(pm);  break;
        case test :  Parameters_changed = TEST.basepair_interaction()->get_potential_parameters(pm); break;
        case pb3DA :  Parameters_changed = PB3DA.basepair_interaction()->get_potential_parameters(pm); break;
        case pcla :  Parameters_changed = PCLA.basepair_interaction()->get_potential_parameters(pm); break;
        case pclj :  Parameters_changed = PCLJ.basepair_interaction()->get_potential_parameters(pm); break;
        case trmf :  Parameters_changed = TRMF.basepair_interaction()->get_potential_parameters(pm); break;
        // *** Add new Hamiltonian here
        // e.g.: case zzzz :  return ZZZZ.stacking_interaction()->get_potential_parameters(pm);
        }
      return Parameters_changed;
      }
      
    friend std::ostream& operator<<(std::ostream &out, const TransferMatrix& tm)
      {
      switch(tm.Hamiltonian_ID)
        {
        case TransferMatrix::pb   :  out << tm.PB;   break;
        case TransferMatrix::dpb  :  out << tm.DPB;  break;
        case TransferMatrix::hms  :  out << tm.HMS;  break;
        case TransferMatrix::jb   :  out << tm.JB;   break;
        case TransferMatrix::mes  :  out << tm.MES;  break;
        case TransferMatrix::test :  out << tm.TEST; break;
        case TransferMatrix::pb3DA :  out << tm.PB3DA; break;
        case TransferMatrix::pcla :  out << tm.PCLA; break;
        case TransferMatrix::pclj :  out << tm.PCLJ; break;
        case TransferMatrix::trmf :  out << tm.TRMF; break;
        // *** Add new Hamiltonian here
        // e.g.: case TransferMatrix::zzzz :  out << tm.ZZZZ; break;
        }
      return out;
      }


    /// \brief Assigns parameters for the Peyrard-Bishop Hamiltonian
    inline bool parameters(const PeyrardBishop<value_type>& p) 
      {
      Parameters_changed=PB.parameters(p);
      Recalculate_diagonalization=Recalculate_diagonalization || Parameters_changed;
      return Parameters_changed;
      }

    /// \brief Assigns parameters for the Daxuois-Peyrard-Bishop Hamiltonian
    inline bool parameters(const DauxoisPeyrardBishop<value_type>& p) 
      {
      Parameters_changed=DPB.parameters(p);
      Recalculate_diagonalization=Recalculate_diagonalization || Parameters_changed;
      return Parameters_changed;
      }

    /// \brief Assigns parameters for the harmonic Morse-Solvent Hamiltonian
    inline bool parameters(const HarmonicMorseSolvent<value_type>& p) 
      {
      Parameters_changed=HMS.parameters(p);
      Recalculate_diagonalization=Recalculate_diagonalization || Parameters_changed;
      return Parameters_changed;
      }

    /// \brief Assigns parameters for the harmonic Morse-Solvent Hamiltonian
    inline bool parameters(const JoyeuxBuyukdagli<value_type>& p) 
      {
      Parameters_changed=JB.parameters(p);
      Recalculate_diagonalization=Recalculate_diagonalization || Parameters_changed;
      return Parameters_changed;
      }

    /// \brief Assigns parameters for the Morse Hamiltonian with Exact Stacking
    inline bool parameters(const MorseExactStacking<value_type>& p) 
      {
      Parameters_changed=MES.parameters(p);
      Recalculate_diagonalization=Recalculate_diagonalization || Parameters_changed;
      return Parameters_changed;
      }

    /// \brief Assigns parameters for the Test Hamiltonian, make sure you modify TestV and TestW in ModelHamiltonian.h 
    inline bool parameters(const TestHamiltonian<value_type>& p) 
      {
      Parameters_changed=TEST.parameters(p);
      Recalculate_diagonalization=Recalculate_diagonalization || Parameters_changed;
      return Parameters_changed;
      }

    inline bool parameters(const PeyrardBishop3DA<value_type>& p) 
      {
      Parameters_changed=PB3DA.parameters(p);
      Recalculate_diagonalization=Recalculate_diagonalization || Parameters_changed;
      return Parameters_changed;
      }

    inline bool parameters(const PeyrardCuestaLopezAngelov<value_type>& p) 
      {
      Parameters_changed=PCLA.parameters(p);
      Recalculate_diagonalization=Recalculate_diagonalization || Parameters_changed;
      return Parameters_changed;
      }

    inline bool parameters(const PeyrardCuestaLopezJames<value_type>& p) 
      {
      Parameters_changed=PCLJ.parameters(p);
      Recalculate_diagonalization=Recalculate_diagonalization || Parameters_changed;
      return Parameters_changed;
      }
      
    inline bool parameters(const TapiaRojoMazoFalo<value_type>& p) 
      {
      Parameters_changed=TRMF.parameters(p);
      Recalculate_diagonalization=Recalculate_diagonalization || Parameters_changed;
      return Parameters_changed;
      }
      
      
      // *** Add new Hamiltonian here
//     inline bool parameters(const XXXX<value_type>& p) 
//       {
//       bool modified=ZZZZ.parameters(p);
//       Recalculate_diagonalization=Recalculate_diagonalization || modified;
//       return modified;
//       }

      
      
    /// \brief Evaluates the K function.
    ///
    /// \attention This method needs to be modified if new Hamiltonians are added.
    inline vector_type K(const vector_type& x, const vector_type& y)
      {
      switch(Hamiltonian_ID)
        {
        case pb   :  return exp(-Beta*PB.evaluate(x,y));
        case dpb  :  return exp(-Beta*DPB.evaluate(x,y));
        case hms  :  return exp(-Beta*HMS.evaluate(x,y));
        case jb   :  return exp(-Beta*JB.evaluate(x,y));
        case mes  :  return exp(-Beta*MES.evaluate(x,y));
        case test :  return exp(-Beta*TEST.evaluate(x,y));
        case pb3DA :  return PB3DA.exp_evaluate3D(x,y,Beta);
        case pcla :  return exp(-Beta*PCLA.evaluate(x,y));
        case pclj :  return exp(-Beta*PCLJ.evaluate(x,y));
        case trmf :  return exp(-Beta*TRMF.evaluate(x,y));
        // *** Add new Hamiltonian here
        // e.g.: case zzzz :  return exp(-Beta*ZZZZ.evaluate(x,y));
        }
      return vector_type();
      }

     /// \brief Evaluates the Y function.
    ///
    /// \attention This method needs to be modified if new Hamiltonians are added.
    inline vector_type basepair_potential(const vector_type& x)
      {
      switch(Hamiltonian_ID)
        {
        case pb   :  return PB.potential()->First.Vx.evaluate(x);
        case dpb  :  return DPB.potential()->First.Vx.evaluate(x);
        case hms  :  return HMS.potential()->First.Vx.evaluate(x);
        case jb   :  return JB.potential()->First.Vx.evaluate(x);
        case mes  :  return MES.potential()->First.Vx.evaluate(x);
        case test :  return TEST.potential()->First.Vx.evaluate(x);
        case pb3DA :  return PB3DA.potential()->First.Vx.evaluate(x);
        case pcla :  return PCLA.potential()->First.Vx.evaluate(x);
        case pclj :  return PCLJ.potential()->First.Vx.evaluate(x);
        case trmf :  return TRMF.potential()->First.Vx.evaluate(x);
        // *** Add new Hamiltonian here
        // e.g.: case zzzz :  return exp(-0.5*Beta*ZZZZ.potential()->First.Vx.evaluate(x));
        }
      return vector_type();
      }
      
    inline vector_type stacking_potential(const vector_type& x, const vector_type& y)
      {
      switch(Hamiltonian_ID)
        {
        case pb   :  PB.stacking_interaction()->evaluate(x,y);
        case dpb  :  DPB.stacking_interaction()->evaluate(x,y);
        case hms  :  HMS.stacking_interaction()->evaluate(x,y);
        case jb   :  JB.stacking_interaction()->evaluate(x,y);
        case mes  :  MES.stacking_interaction()->evaluate(x,y);
        case test :  TEST.stacking_interaction()->evaluate(x,y);
        case pb3DA :  PB3DA.stacking_interaction()->evaluate(x,y);
        case pcla :  PCLA.stacking_interaction()->evaluate(x,y);
        case pclj :  PCLJ.stacking_interaction()->evaluate(x,y);
        case trmf :  TRMF.stacking_interaction()->evaluate(x,y);
        // *** Add new Hamiltonian here
        // e.g.: case zzzz :  return exp(-Beta*ZZZZ.basepair_interaction()->evaluate(x,y));
        }
      return vector_type();
      }


   /// \brief Evaluates the Y function.
    ///
    /// \attention This method needs to be modified if new Hamiltonians are added.
    inline vector_type Y(const vector_type& x)
      {
      switch(Hamiltonian_ID)
        {
        case pb   :  return exp(-0.5*Beta*PB.potential()->First.Vx.evaluate(x));
        case dpb  :  return exp(-0.5*Beta*DPB.potential()->First.Vx.evaluate(x));
        case hms  :  return exp(-0.5*Beta*HMS.potential()->First.Vx.evaluate(x));
        case jb   :  return exp(-0.5*Beta*JB.potential()->First.Vx.evaluate(x));
        case mes  :  return exp(-0.5*Beta*MES.potential()->First.Vx.evaluate(x));
        case test :  return exp(-0.5*Beta*TEST.potential()->First.Vx.evaluate(x));
        case pb3DA :  return sqrt(x) * exp(-0.5*Beta*PB3DA.potential()->First.Vx.evaluate(x));
        case pcla :  return exp(-0.5*Beta*PCLA.potential()->First.Vx.evaluate(x));
        case pclj :  return exp(-0.5*Beta*PCLJ.potential()->First.Vx.evaluate(x));
        case trmf :  return exp(-0.5*Beta*TRMF.potential()->First.Vx.evaluate(x));
        // *** Add new Hamiltonian here
        // e.g.: case zzzz :  return exp(-0.5*Beta*ZZZZ.potential()->First.Vx.evaluate(x));
        }
      return vector_type();
      }

    /// \brief Evaluates the A function.
    ///
    /// \attention This method needs to be modified if new Hamiltonians are added.
    inline vector_type A(const vector_type& x, const vector_type& y)
      {
      switch(Hamiltonian_ID)
        {
        case pb   :  return exp(-Beta*PB.basepair_interaction()->evaluate(x,y));
        case dpb  :  return exp(-Beta*DPB.basepair_interaction()->evaluate(x,y));
        case hms  :  return exp(-Beta*HMS.basepair_interaction()->evaluate(x,y));
        case jb   :  return exp(-Beta*JB.basepair_interaction()->evaluate(x,y));
        case mes  :  return exp(-Beta*MES.basepair_interaction()->evaluate(x,y));
        case test :  return exp(-Beta*TEST.basepair_interaction()->evaluate(x,y));
        case pb3DA :  return exp(-Beta*PB3DA.basepair_interaction()->evaluate(x,y));
        case pcla :  return exp(-Beta*PCLA.basepair_interaction()->evaluate(x,y));
        case pclj :  return exp(-Beta*PCLJ.basepair_interaction()->evaluate(x,y));
        case trmf :  return exp(-Beta*TRMF.basepair_interaction()->evaluate(x,y));
        // *** Add new Hamiltonian here
        // e.g.: case zzzz :  return exp(-Beta*ZZZZ.basepair_interaction()->evaluate(x,y));
        }
      return vector_type();
      }

    /// \brief Sets the new temperature.
    ///
    /// Also recalculates the Beta value.
    void temperature(double T) 
      {
      if (T != Temperature) Recalculate_diagonalization=true;
      Temperature=T; 
      Beta=1.0/(GSL_CONST_MKSA_BOLTZMANN/GSL_CONST_MKSA_ELECTRON_VOLT*T);
      }

    void calculate_eigenvalue_only(bool ceo=true) {eigen_engine.Calculate_eigenvalue_only=ceo;}

    /// \brief Sets the limits for the integration
    ///
    /// This method recalculated the integration weights and rescales the integration
    /// space if necessary.
    void limits(value_type l, ///< Lower integration limit, usually negative 
                value_type u, ///< Upper integration limit
                size_t s)     ///< Number of integration points
      {
      bool recalculate=false;
      if (Integration_steps != s)
        {
        Integration_steps=s;
        Integration_space.resize(Integration_steps);
        Weights.resize(Integration_steps);
        W.resize(Integration_steps,Integration_steps);
        Rsqrt.resize(Integration_steps,Integration_steps);
        Rminus_sqrt.resize(Integration_steps,Integration_steps);
        YD.resize(Integration_steps,Integration_steps);
        tmp_xy.resize(Integration_steps,Integration_steps);
        recalculate=true;
        SqrtWeights.resize(Integration_steps);
        size_t N;
        if (Eigenvalue_cutoff > (size_t)0) N=Eigenvalue_cutoff;
        else N=Integration_steps;
        phi.resize(Integration_steps,N);
        }
      if ((Lower_limit !=l) || (Upper_limit!=u))
        {
        Lower_limit=l; Upper_limit=u; 
        recalculate=true;
        }
      if (recalculate)
        {
        Integration_space=linear_space<vector_type>(Lower_limit,Upper_limit,Integration_steps);
        quadrature->adjust(Integration_space);
        Weights=quadrature->Rule.weights();
        W.diagonal(Weights);
        YD.diagonal(Integration_space);
        SqrtWeights=sqrt(Weights);
        Rsqrt.diagonal(SqrtWeights);
        Rminus_sqrt.diagonal(pow(SqrtWeights,value_type(-1.0)));
        Recalculate_diagonalization=true;
        }
      }

    /// \brief Executes the calculation.
    void diagonalize(void)
      {
      if (Recalculate_diagonalization)
        {
        tmp_xy=K(Integration_space,Integration_space);

        tmp_xy=Rsqrt*tmp_xy*Rsqrt; //wiki:Projeto/Termodinâmica/Modelo Peyrard-Bishop/Desenvolvimento#Discretização

        CERR_DDEBUG(DTRM_DIAGNAL) << " diagonalizing" << std::endl;

        eigen_engine.diagonalize(tmp_xy);

        if (!eigen_engine.Calculate_eigenvalue_only) 
          {
          eigen_engine.Eigenvector=Rminus_sqrt*eigen_engine.Eigenvector;
          normalize_eigenvectors();
          save_matrix(eigen_engine.Eigenvector,"eigenvectors");
          }
        Recalculate_diagonalization=false;
        }
      save_matrix(eigen_engine.Eigenvalue,"eigenvalues");
      }

    /// \brief Calculates the matrix \f$ C_{n,m}\f$
    matrix_type C_matrix_calculation(const matrix_type& base)
      {
      size_t N;
      if (Eigenvalue_cutoff > (size_t)0) N=Eigenvalue_cutoff;
      else N=Integration_steps;
      if (phi.Columns != N) phi.resize(Integration_steps,N);

      phi=base.submatrix(0,0,Integration_steps,N);

      tmp_xy=K(Integration_space,Integration_space);

      return transpose(phi)*W*tmp_xy*W*phi;
      };
      
    /// \brief Calculates the matrix \f$ C_{n,m}\f$
    void AC_matrix_calculation(const matrix_type& base, matrix_type& Amatrix, matrix_type& Cmatrix)
      {
      size_t N;
      if (Eigenvalue_cutoff > (size_t)0) N=Eigenvalue_cutoff;
      else N=Integration_steps;
      if (phi.Columns != N) phi.resize(Integration_steps,N);

      phi=base.submatrix(0,0,Integration_steps,N);
      
      matrix_type tphiW=transpose(phi)*W, Wphi=W*phi;

      tmp_xy = K(Integration_space,Integration_space);
      Cmatrix = tphiW*tmp_xy*Wphi;

      tmp_xy = A(Integration_space,Integration_space);
      Amatrix = tphiW*tmp_xy*Wphi;
      };

    /// \brief Calculates the matrix \f$ C_{n,m}\f$
    matrix_type A_matrix_calculation(const matrix_type& base)
      {
      size_t N;
      if (Eigenvalue_cutoff > (size_t)0) N=Eigenvalue_cutoff;
      else N=Integration_steps;
      if (phi.Columns != N) phi.resize(Integration_steps,N);

      phi=base.submatrix(0,0,Integration_steps,N);
      tmp_xy=A(Integration_space,Integration_space);

      return transpose(phi)*W*tmp_xy*W*phi;
      };

    /// \brief Calculates the matrix \f$ Y_{n,m}\f$
    matrix_type Y_matrix_calculation(const matrix_type& base)
      {
      size_t N;
      if (Eigenvalue_cutoff > (size_t)0) N=Eigenvalue_cutoff;
      else N=Integration_steps;
      if (phi.Columns != N) phi.resize(Integration_steps,N);

      phi=base.submatrix(0,0,Integration_steps,N);

      return transpose(phi)*YD*W*phi;
      };



    /// \brief Performs the eigenvector normalization.
    void normalize_eigenvectors(void)
      {
      size_t l;
      for (l=0; l < Integration_steps; ++l)
        {
        vector_type eig=eigenvector(l);
        double norm=vector_type(pow(eig,value_type(2.0))*Weights).sum();
        eig /= sqrt(norm);
        eigen_engine.Eigenvector.column(l,eig);
        }
      }

    /// \brief Retrieves the n-th calculated eigenvalue. 
    inline value_type eigenvalue(const size_t& n) const
      {return eigen_engine.Eigenvalue[n];}

    /// \brief Retrieves the n-th calculated eigenvalue. 
    inline vector_type eigenvalues(void) const
      {return eigen_engine.Eigenvalue.Array;}

    /// \brief Retrieves the n-th calculated eigenvector. 
    inline vector_type eigenvector(const size_t& n) const
      {return eigen_engine.Eigenvector.column(n).Array;}

    /// \brief Retrieves the n-th calculated eigenfunction. 
    inline value_type eigenfunction(const value_type& x, const size_t& n)
      {
      vector_type xv(x,1);
      CERR_DDEBUG(DTRM_VECKXV) << vector_type(K(xv,Integration_space)).sum() << " " << std::endl;
      vector_type kxy(1,Integration_steps);
      kxy=K(xv,Integration_space);
      return vector_type(Weights*kxy*eigenvector(n)).sum()/eigenvalue(n);
      }

    /// \brief Calculates the partition function \f$Z\f$ for \f$N\f$ base-pairs.
    ///
    /// \f$ Z=\sum_{n=1}^{\infty}\lambda_n^N \f$
    value_type partition_function(const unsigned int& N) const
      {
      vector_type z=pow(eigenvalues(),static_cast<value_type>(N));
      return z.sum();
      }

    /// \brief Calculates the coefficient \f$a_m\f$ needed for open boundary functions.
    ///
    /// \f$ a_m=\int \phi_m(y)e^{-\frac{1}{2}\beta V(y)}dy \f$
    double coef_am(const size_t& m)
      {
      return vector_type(eigenvector(m)*Y(Integration_space)*Weights).sum();
      }

    /// \brief Calculates \f$\lange n| y| m\angle$\f.
    /// See zhang97b Eq. 13
    value_type average_y(const size_t& n,const size_t& m) const
      {
      return vector_type(eigenvector(n)*Integration_space*eigenvector(m)*Weights).sum();
      }

    /// \brief Calculates \f$\lange y \angle$\f.
    ///
    /// \f$ \langle y \rangle 
    /// =\frac{1}{Z_y}\sum_{n=1}^{\infty}\lambda_n^N\langle n|y|n\rangle \f$
    /// See zhang97b Eq. 13
    value_type average_y(const unsigned int& N) ///< Number of base pairs (starts at 1)
      const
      {
      vector_type z=pow(eigenvalues(),static_cast<value_type>(N))/partition_function(N);
      size_t n, mx=Eigenvalue_cutoff;
      if (Eigenvalue_cutoff == 0) mx=eigenvalues().size();
      value_type y=0.0;
      for (n=0; n < mx; ++n)
         {
         y += z[n]*average_y(n,n);
         }
      return y;
      }

    /// \brief Calculates \f$ \langle | y_m | \rangle \f$ with open boundary conditions.
    /// \f$ \langle y_m \rangle=
    /// \frac{\sum_{l,n}a_la_n\Delta_l^{N-m}\Delta_n^{m-1}\langle n|y_m|l\rangle}{\sum_n\Delta_n^{N-1}a_n^2}
    /// \f$
    vector_type open_average_y(const unsigned int& N) ///< Number of base pairs (starts at 1)
      {
      size_t n, cutoff=Eigenvalue_cutoff;
      if (Eigenvalue_cutoff == 0) cutoff=eigenvalues().size();
      vector_type Delta(cutoff);
      Delta=vector_type(eigen_engine.Eigenvalue.Array[std::slice(0,cutoff,1)])/eigenvalue(1);
      vector_type a(cutoff);
      double s1=0.0;
      vector_type s2(N);
      for (n=0; n < cutoff; ++n) 
        {
        a[n]=coef_am(n);
        s1+=pow(Delta[n],static_cast<value_type>(N-1))*pow(a[n],2.0);
        }
      for (size_t m=1; m <= N; ++m)
        {
        s2[m-1]=0.0;
        for (n=0; n < cutoff; ++n)
          {
          double n_only=a[n]*pow(Delta[n],static_cast<value_type>(m-1))/s1;
          for (size_t l=0; l < cutoff; ++l)
            s2[m-1]+=n_only*a[l]*pow(Delta[l],static_cast<value_type>(N-m))*average_y(n,l);
          }
        }
      return s2;
      }

    std::string matrix_filename(std::string mat_name)
      {
      std::stringstream filename;
      filename << MatrixDir << "/" << ParameterIdName
               << "-" << Temperature << "-" << mat_name;
      return filename.str();
      }

    /// \brief Stores matrices into the map and saves into a directory.
    ///
    /// \attention If you don't want to store the matrix in a file
    /// leave MatrixDir empty.
    void save_matrix(const matrix_type& mat,
                     std::string mat_name)
      {
      if (!MatrixDir.empty())
        {
        boost::filesystem::create_directory(MatrixDir);
        std::ofstream outfile;
        std::string filename;
        filename=matrix_filename(mat_name);
        outfile.open(filename.c_str());
        outfile << *this << std::endl;
        outfile << mat;
        outfile.close();
        quadrature->Rule.Base_directory=MatrixDir;
        }
      }

   };

 };

#endif
