// $Id: mesoscopic-potentials.cpp 1317 2022-09-22 20:13:52Z ge $
/// \file tfreg.cpp
/// \brief Calculates temperatures using the regression method
/// \author Gerald Weber <gweberbh@gmail.com>
/// $Revision: 1317 $

#include "Options.h"
#include "NeighbourSequence.h"
#include "Regression.h"
#include <unistd.h>
#include "HeterogenousTM.h"
//#include "Actions.h"
#include "RegressionModel.h"
#include "CPUtime.h"
#include "OutStream.h"
#include "ErrorCodes.h"

namespace gbc
  {
  template<>
  std::valarray<HeterogenousTM<>::value_type> ActionAverageY::action(HeterogenousTM<>& tf,size_t N)
    {
    number_of_basepairs=tf.pSI->BP_reduced_duplex.size();
    tf.retrieve_eigensystem();
    tf.retrieve_or_calculate_matrices();
    return tf.average_y();
    }
  };

using namespace gbc;
using namespace std;

typedef double value_type;

int main(int argc,char *argv[])
  {
  
#ifndef SVNVERSION
#define SVNVERSION "$Revision: 1317 $"
#endif
#ifndef LINUXSYSTEM
#define LINUXSYSTEM "UNKNOWN"
#endif
#ifndef BUILDTIME
#define BUILDTIME "UNKNOWN"
#endif
#ifndef BUILDTIME
#define BUILDTIME "UNKNOWN"
#endif
#ifndef COMPILERNAME
#ifdef __INTEL_COMPILER
#define COMPILERNAME "icpc"    
#else
#define COMPILERNAME "g++"
#endif
#endif

  OptionMap::env_id="MESPOT";
   
  std::cout << "*** mesoscopic-potentials version=" << std::string(SVNVERSION) << " system=" << std::string(LINUXSYSTEM) << " builddate=" << std::string(BUILDTIME) << " compiler=" << std::string(COMPILERNAME) << std::endl;

 
  Option<string>         output_name("o","output files basename",OptionMap::necessary);
  
  Option<string>         result("res","select which type of results",OptionMap::optional,"bp");
  result.add_restricted_option("bp","base pair potentials");
  result.add_restricted_option("stacking","nearest-neighbour stacking potentials");
  
  Option<deque<string> > par_file("par","list of parameter file",OptionMap::necessary);
  Option<string>         bp1("bp1","base pair 1",OptionMap::optional,"CG");
  Option<string>         bp2("bp2","base pair 2",OptionMap::optional,"CG");
  
  Option<deque<string> > debug("debug","select debuging flags",OptionMap::optional);

  Option<std::string>    model("model","model",OptionMap::optional,"pb");
  model.add_restricted_option("pb","Peyrard-Bishop model");
  model.add_restricted_option("dpb","Dauxois-Peyrard-Bishop  model");
  model.add_restricted_option("hms","Harmonic-Morse-Solvent model");
  model.add_restricted_option("jb","finite enthalpy model");
  model.add_restricted_option("mes","Morse-exact stacking model");
  model.add_restricted_option("test","Reserved");
  model.add_restricted_option("pb3DA","approximated 3D Peyrard-Bishop");
  model.add_restricted_option("pcla","entropic barrier y³");
  model.add_restricted_option("pclj","entropic hump potential");
  model.add_restricted_option("trmf","gaussian barrier");
  
  Option<Range<double> > y1("y1","y1",OptionMap::necessary);
  Option<Range<double> > y2("y2","y2",OptionMap::optional);
    
  Option<map<string,string> > dict("dict","Additions to nucleotide dictionary",OptionMap::optional);
    
  Option<bool>           printusedpar("printusedpar","print parameters that were used with file extension .usedpar",OptionMap::optional,false);


  OptionMap::arguments(argc,argv);
  OptionMap::scan_arguments();
  
  if (result.check("stacking"))
    {
    bp2.turn_necessary();
    y2.turn_necessary();
    }

  OptionMap::check_if_all_given();
  OptionMap::show_options();

  if (debug.provided()) 
    for(auto db : debug.Value) DebugControl::enable(db);
      
  if (dict.provided())
    {
    for(auto mit : dict.Value)
      {
      std::cout << "Adding new nucleotide to dictionary= "  << mit.first[0] << " and its complementary= " << mit.second[0] << std::endl;
      Nucleotide<>::dictionary.add(mit.first[0],mit.first[0],true);
      Nucleotide<>::dictionary.add(mit.second[0],mit.second[0],true);
      Nucleotide<>::dictionary.complementary_pair(mit.second[0],mit.first[0]);
      }
    }

  BasePairNeighbours<> bpn(bp1.Value,bp2.Value);

  ParameterMap<> parameter_map;
  if (par_file.provided())
     parameter_map.read_from_list_of_files(par_file.Value);

  TransferMatrix<> TM(bpn);
  TM.select_hamiltonian((std::string)model);
  TM.get_hamiltonian_parameters(parameter_map);


  //Defines all files, they will be created on first direct write or explicit stream_open
  OutStream outfile((string)output_name + "|dat"); 
  OutStream used_par_file((string)output_name + "|usedpar"); 
  
  std::valarray<double> xvec=y1.Value.convert2valarray();
  std::valarray<double> yvec=y2.Value.convert2valarray();
  
  if (result.check("bp"))
    {
  
    std::valarray<double> Vx=TM.basepair_potential(xvec);
  
    for(int i=0; i < xvec.size(); i++)
      outfile << " " << xvec[i] << " " << Vx[i] << std::endl;
    }

  if (result.check("stacking"))
    {
  
    std::valarray<double> Vxy=TM.stacking_potential(xvec,yvec);
  
    int nx=xvec.size() , ny=yvec.size();
  
     for(int i=0; i < xvec.size(); i++)
       {
       for (int j=0; j < yvec.size(); j++) outfile << Vxy[i*ny+j] << " ";
       outfile << std::endl;
       }
    }
    
    
  if (printusedpar.check(true))
    {
    used_par_file.stream_open();
    parameter_map.print_parameters_and_stats(used_par_file,"parameters");
    used_par_file.stream_close();
    }
    
  outfile.stream_close();
  print_cpu();

  return 0;
}

