// $Id: Hamiltonian.h 1252 2021-12-20 14:43:47Z ge $
/// \file Hamiltonian.h
/// \brief contains the Hamiltonian function class
///
/// $Revision: 1252 $
/// \author Gerald Weber <gweberbh@gmail.com>
#ifndef GBC_HAMILTONIAN_H
#define GBC_HAMILTONIAN_H "$Id: Hamiltonian.h 1252 2021-12-20 14:43:47Z ge $"
#include <valarray>
#include "BasePairNeighbours.h"
#include "ModelPotentials.h"
namespace gbc
  {
  template<class _BasePair, class _Stacking>
  class Hamiltonian: 
    public CmbPotential<SymmPotential<_BasePair>,_Stacking>
    {
    public:
      typedef CmbPotential<SymmPotential<_BasePair>,_Stacking> base_type;
      typedef SymmPotential<_BasePair> basepair_type;
      typedef _Stacking stacking_type;
      typedef typename _Stacking::value_type value_type;

    Hamiltonian(void): base_type() {}

    /// \brief Constructor that takes the stacked nearest neighbours string.
    ///
    /// This constructor takes as string like "AT_GC" and splits it into
    /// two base pairs by using the smallest symmetry for the base_pairs, in this
    /// case we would have "AT" and "CG".
    Hamiltonian(const BasePairNeighbours<>& stk) ///< stk is a string representing the stacked nearest neighbours, e.g. "AT_CG"
      : base_type(stk.first,stk.second,stk) {}

    inline base_type*     potential(void) {return this;}
    inline basepair_type* basepair_interaction(void) {return &(this->First);}
    inline stacking_type* stacking_interaction(void) {return &(this->Second);}
    
    // Checks if Vx parameters equals Vy and vice-versa, but stacking needs to be the same
    inline bool switched_basepair_same_stacking(const Hamiltonian &ham) const 
      {
      return this->First.switched_par_values(ham.First) and this->Second.same_par_values(ham.Second);
      } 

    
    friend bool same_parameters(const Hamiltonian &h1, const Hamiltonian &h2)
      {
      return     h1.Second.ParMap   == h2.Second.ParMap
             and h1.First.Vx.ParMap == h2.First.Vx.ParMap
             and h1.First.Vy.ParMap == h2.First.Vy.ParMap;
      }

    };

  template<class _Tp=double>
  class PeyrardBishop: public Hamiltonian<Morse<_Tp>,HarmonicStacking<_Tp> >
    {
    public: 
      typedef _Tp value_type;
      typedef Hamiltonian<Morse<_Tp>,HarmonicStacking<_Tp> > base_type;

    PeyrardBishop(void): base_type() {}

    PeyrardBishop(const BasePairNeighbours<>& stk)
      : base_type(stk) {}

    };

  //dauxois93 doi:10.1103/PhysRevE.47.R44
  template<class _Tp=double>
  class DauxoisPeyrardBishop: public Hamiltonian<Morse<_Tp>,AnharmonicStacking<_Tp> >
    {
    public: 
      typedef _Tp value_type;
      typedef Hamiltonian<Morse<_Tp>,AnharmonicStacking<_Tp> > base_type;

    DauxoisPeyrardBishop(void): base_type() {}

    DauxoisPeyrardBishop(const BasePairNeighbours<>& stk)
      : base_type(stk) {}

    };
    
  //peyrard09 doi:10.1088/0953-8984/21/3/034103
  template<class _Tp=double>
  class PeyrardCuestaLopezAngelov: public Hamiltonian<MorseBarrierY3<_Tp>,AnharmonicStacking<_Tp> >
    {
    public: 
      typedef _Tp value_type;
      typedef Hamiltonian<MorseBarrierY3<_Tp>,AnharmonicStacking<_Tp> > base_type;

    PeyrardCuestaLopezAngelov(void): base_type() {}

    PeyrardCuestaLopezAngelov(const BasePairNeighbours<>& stk)
      : base_type(stk) {}

    };
    
  //peyrard09b doi:10.1007/s10867-009-9127-2
  template<class _Tp=double>
  class PeyrardCuestaLopezJames: public Hamiltonian<Hump<_Tp>,AnharmonicStacking<_Tp> >
    {
    public: 
      typedef _Tp value_type;
      typedef Hamiltonian<Hump<_Tp>,AnharmonicStacking<_Tp> > base_type;

    PeyrardCuestaLopezJames(void): base_type() {}

    PeyrardCuestaLopezJames(const BasePairNeighbours<>& stk)
      : base_type(stk) {}

    };

  //tapiarojo10 doi:10.1103/PhysRevE.82.031916
  template<class _Tp=double>
  class TapiaRojoMazoFalo: public Hamiltonian<MorseGaussian<_Tp>,AnharmonicStacking<_Tp> >
    {
    public: 
      typedef _Tp value_type;
      typedef Hamiltonian<MorseGaussian<_Tp>,AnharmonicStacking<_Tp> > base_type;

    TapiaRojoMazoFalo(void): base_type() {}

    TapiaRojoMazoFalo(const BasePairNeighbours<>& stk)
      : base_type(stk) {}

    };
    
  //weber06b doi:10.1209/epl/i2005-10466-6
  template<class _Tp=double>
  class HarmonicMorseSolvent: public Hamiltonian<MorseSolvent<_Tp>,HarmonicStacking<_Tp> >
    {
    public: 
      typedef _Tp value_type;
      typedef Hamiltonian<MorseSolvent<_Tp>,HarmonicStacking<_Tp> > base_type;

    HarmonicMorseSolvent(void): base_type() {}

    HarmonicMorseSolvent(const BasePairNeighbours<>& stk)
      : base_type(stk) {}

    };

  template<class _Tp=double>
  class JoyeuxBuyukdagli: public Hamiltonian<Morse<_Tp>,FiniteEnthalpyStacking<_Tp> >
    {
    public: 
      typedef _Tp value_type;
      typedef Hamiltonian<Morse<_Tp>,FiniteEnthalpyStacking<_Tp> > base_type;

    JoyeuxBuyukdagli(void): base_type() {}

    JoyeuxBuyukdagli(const BasePairNeighbours<>& stk)
      : base_type(stk) {}

    };
    
  template<class _Tp=double>
  class MorseExactStacking: public Hamiltonian<Morse<_Tp>,ExactStacking<_Tp> > // Here specify the how the Hamiltonian is build, the order is important
    {
    public: 
      typedef _Tp value_type;
      typedef Hamiltonian<Morse<_Tp>,ExactStacking<_Tp> > base_type;// Here specify the how the Hamiltonian is build, the order is important

    MorseExactStacking(void): base_type() {}

    MorseExactStacking(const BasePairNeighbours<>& stk)
      : base_type(stk) {}

    };
    
  template<class _Tp=double>
  class PeyrardBishop3DA: public Hamiltonian<Morse3D<_Tp>,HarmonicStacking3DA<_Tp> >
    {
    public: 
      typedef _Tp value_type;
      typedef Hamiltonian<Morse3D<_Tp>,HarmonicStacking3DA<_Tp> > base_type;

    PeyrardBishop3DA(void): base_type() {}

    PeyrardBishop3DA(const BasePairNeighbours<>& stk)
      : base_type(stk) {}

    };

    
  };

#endif
