// $Id: IntegerFormat.h 967 2017-07-13 12:04:25Z ge $
/// \file IntegerFormat.h
/// \brief Contains the base class IntegerFormat
///
/// $Revision: 967 $
/// \author Gerald Weber <gweberbh@gmail.com>
#ifndef GBC_INTEGERFORMAT_H
#define GBC_INTEGERFORMAT_H "$Id: IntegerFormat.h 967 2017-07-13 12:04:25Z ge $"
#include "FileFormat.h"

namespace gbc {

/// \brief Parses a file in Integer format.
///
/// This class is written for fast reading of Nucleotide Files in integer format.
/// It does not map contigs.
/// \attention This class by-passes NucleotideDictionary when reading files, if you
/// have Nucleotides any other than A,C,G and T do not use this format. 
class IntegerFormat: public FileFormat
  {
  public:

  /// The void constructor.
  IntegerFormat(void) 
    {
    FileFormat::contig_capable=false;
    }

  /// Constructor which takes the filename as argument.
  IntegerFormat(const std::string &fn): FileFormat(fn) 
    {
    FileFormat::contig_capable=true;
    }

  /// Verifies if the file starts with a 0 as integer files do.
  virtual bool check_file_type(void)
    {
    char cr; int intval=1; bool is_int=false;
    rewind();
    file_handle >> cr; intval=cr-48;
    if (intval==0) {is_int=true;}
    if (is_int && Verbose) {std::cout << "Integer format detected." << std::endl;}
    return is_int;
    }

  /// Prepares to read by simply going to the begining of the file.
  inline virtual void prepare_reading(void)
    {
    if (forward_reading) rewind();
    else
      {
      forward_reading=true;
      map_contigs();
      forward_reading=false;
      file_handle.clear();
      file_handle.seekg(-1,std::ios::end);
      if (file_handle.peek() == '\n' && file_handle.tellg() != (std::streampos)0) file_handle.seekg(-1,std::ios_base::cur);
      }
    }

  /// Maps all contigs in a file and places them into FileParser::Contig_table.
  virtual void map_contigs(void) {};

  /// brief Specialized function for reading files in fasta format.
  ///
  /// If the file has been mapped for contigs only the sections between contigs are
  /// read.
  virtual inline void read(Nucleotide<> &nt) ///< Nucleotide to be assigned to
    {integer_read(nt);}
    
  template<class _Tp>
  /// \brief Function for reading files in integer format.
  inline void integer_read(Nucleotide<_Tp> &nt) ///< Nucleotide to be assigned to
    {
    Nucleotide<unsigned int> nti; integer_read(nti);
    nt=nti;
    }

  /// \brief Specialized function for reading with unsigned int
  inline void integer_read(Nucleotide<unsigned int> &nt) ///< Nucleotide to be assigned to
    {
    nt.Good=Contig_begin=false;
    char cr;
    while (!eof())
      {
      file_handle >> cr; int res=cr-48;
      nt.representation=static_cast<unsigned int>(res);
      if (eof()) break;
      if(nt.representation)
        {
        if (get_complementary) nt.complementary();
        nt.Good=true;
        break;
	}
      else  {Contig_begin=true;}
      }
    }

  /// \brief Specialized function for reading with unsigned short int
  inline void integer_read(Nucleotide<unsigned short int> &nt) ///< Nucleotide to be assigned to
    {
    nt.Good=Contig_begin=false;
    char cr;
    while (!eof())
      {
      file_handle >> cr; int res=cr-48;
      nt.representation=static_cast<unsigned short int>(res);
      if (eof()) break;
      if(nt.representation)
        {
        if (get_complementary) nt.complementary();
        nt.Good=true;
        break;
	}
      else  {Contig_begin=true;}
      }
    }

 };
};

#endif






