// $Id: JobControl.h 628 2011-08-31 15:42:16Z ge $
/// \file JobControl.h
/// \brief contains the JobControl class
///
/// $Revision: 628 $
/// \author Gerald Weber <gweberbh@gmail.com>

#ifndef GBC_JOBCONTROL_H
#define GBC_JOBCONTROL_H "$Id: JobControl.h 628 2011-08-31 15:42:16Z ge $"

#include <sys/types.h>
#include <unistd.h>
#include <sys/resource.h>
#include <utility>
#include <string>
#include <map>
#include <set>

namespace gbc {

/// \brief JobControl handles information about the process and provides means to control it.
///
/// All members of this class are static, meaning that you never have to create an object
/// of type JobControl. E.g., to get the amount of used memory simply write JobControl::memory();
/// This class also handles cvs version information.
///
class JobControl
  {
  public:
//    static bool CVS_info;                   ///< This is only to force cvs info initialization in module.
    static bool JobControlEnabled;     ///< Flag which, if set to false, disables JobControlCompletely;
    static std::ofstream LogFile;      ///< Output stream to which events are logged.
    static std::ofstream StatusFile;   ///< Output stream to which current status is written.
  public:
    typedef std::map<std::string,long int> int_map_type;
    typedef std::map<std::string,std::string> string_map_type;
    typedef std::set<std::string> version_set_type;
    static pid_t Pid;                  ///< The process identification.
    static pid_t ParentPid;            ///< The parent process identification.
    static std::string ExecutableFile; ///< The filename of the executable, in parentheses.
    static std::string Username;       ///< Username.
    static char State;                 ///< One character from the string "RSDZTW" where R is running, S is sleeping in an interruptible wait, D is waiting in uninterruptible disk sleep, Z is zombie, T is traced or stopped (on a signal), and W is paging.
    static long int Pagesize;          ///< Size of one page of memory, usually 4k.
    static long int Pages;             ///< Memory used by the process measured in pages (see Pagesize).
    static long int MaxPages;          ///< Maximum amount of memory used by the process measured in pages (see Pagesize).
    static long int Cputime;           ///< CPU time in seconds/100.
    static std::ifstream StatFile;     ///< Status file, usually /proc/<pid number>/stat
    static std::ifstream StatmFile;    ///< Memory status file, usually /proc/<pid number>/statm
    static std::string Proc_path;      ///< Path of the proc file for the current process
    static std::string LogDir;         ///< Directory where we store log and status files.
    static std::string LogFileName;
    static std::string StatusFileName;
    static int_map_type status_int_map;         ///< Map of int status flags.
    static string_map_type status_string_map;   ///< Map of string status flags.
    static string_map_type cpu_string_map;      ///< Map of string status flags.
    static version_set_type cvs_info;           ///< Set of cvs information
    static std::string Hostname;       ///< Hostname where current process is running.
    static std::string Osrelease;      ///< Release version of the operating system
    static long int MaxPhysicalMemory; ///< Maximal physical memory detected for this machine.

  /// \brief Void constructor. Does absolutely nothing.
  JobControl(void);

  /// \brief Disables JobControl writing to files
  static void disable(void);

  /// \brief Enables JobControl writing to files
  static void enable(void);

  /// \brief Works out what the /proc/<pid number> path of the current process is.
  static std::string proc_path(void);

  /// \brief Reads the first line of a /proc file and returns it as a string.
  static std::string read_proc_file(std::string filename);
  
  /// \brief Works out what the /proc/<pid number>/stat path of the current process is.
  static std::string stat_path(void);

  /// \brief Works out what the /proc/<pid number>/statm path of the current process is.
  static std::string statm_path(void);

  /// \brief Creates the log directory and returns it's name.
  static std::string make_log_dir(void);

  /// \brief Works out what the log file path should be.
  static std::string log_path(std::string ext=std::string(".log"));

  /// \brief Queries /proc/<pid number>/statm about the number of pages used by this process.
  ///
  /// Every time this function is called we update MaxPages as well.
  static long int pages(void);

  /// \brief Returns the maximum amount of pages used.
  ///
  /// The accuracy of this function depends on how many times you call pages or memory functions.
  static long int max_pages(void);

  /// \brief Works out the difference between the current number of pages and the last time any memory function was called.
  ///
  /// Any call to functions memory(), diff_memory(), pages() and diff_pages() resets the counter.
  static long int diff_pages(void);

  /// \brief Queries /proc/<pid number>/statm about the memory used by this process.
  ///
  /// It is the same as pages()*Pagesize.
  static long int memory(void);

  /// \brief Returns the maximum amount of memory used.
  ///
  /// It is the same as max_pages()*Pagesize.
  static long int max_memory(void);

  /// \brief Returns the proportion of Physical Memory in use (from 0 to 1000).
  static int physical_memory_in_use(void);

  /// \brief Works out the difference between the current memory and the last time any memory function was called.
  ///
  /// Any call to functions memory(), diff_memory(), pages() and diff_pages() resets the counter.
  static long int diff_memory(void);

  /// \brief Reads the size of the physical memory, value in Pages.
  static long int read_meminfo(void);

  /// \brief Reads the contents of /proc/<pid number>/stat
  static void read_stat(void);

  /// \brief Returns the cputime used by this process.
  static long int  cputime(void);

  static long int  diff_cputime(void);

  /// \brief Writes a log entry.
  static void write_log(std::string event=std::string()); ///< The event description.

  /// \brief Writes a process status.
  static void write_status(void); ///< The event description.

  /// \brief Record the status of a given integer flag.
  static void status(std::string desc, ///< Status variable, will be created if it does not exist
                     int stat);        ///< Variable value.

   /// \brief Record the status of a given string flag.
  static void status(std::string desc,  ///< Status variable, will be created if it does not exist
                     std::string stat); ///< Variable value.

  /// \brief Records cvs information.
  static bool cvs(std::string header,                ///< CVS information
                  std::string module=std::string()); ///< Additional CVS information

  /// \brief Writes all information gathered with cvs().
  static std::ostream& cvs(std::ostream& fl); ///< The output stream.

  /// \brief Writes the job summary.
  ///
  /// You would call this typically at the end of your program execution.
  static void summary(void);
  static std::ostream& summary(std::ostream& fl);

  static std::ostream& show_file_names(std::ostream& fl);
  };
};

#endif
  
