// $Id: SequenceSet.h 697 2012-11-05 15:57:05Z ge $
/// \file SequenceSet.h
/// \brief Contains the definition of the class SequenceSet.
///
/// $Revision: 697 $
/// \author Gerald Weber <gweberbh@gmail.com>
#ifndef SEQUENCESET_H
#define SEQUENCESET_H "$Id: SequenceSet.h 697 2012-11-05 15:57:05Z ge $"
#include <set>
#include <iterator>
#include "NucleotideSequence.h"

namespace gbc {
template<class _Sequence=NucleotideSequence<>,   ///< The type of sequence to be used.
         class _Container=std::set<_Sequence> >  ///< The type of container to be used.
/// \brief Set (std::set) of NucleotideSequences.
///
/// Holds a set of NucleotideSequences in a std::set.
class SequenceSet:public _Container
  {
  public:
  typedef SequenceSet sequenceset_type;
  typedef _Sequence  sequence_type;                 ///< type of _Sequence
  typedef _Container container_type;                ///< type of _Container
  sequence_type DefaultNS;                          ///< an object of type NucleotideSequence to get the default values
  std::string Separator;                            ///< string that separates sequences on output.

  /// \brief The void constructor.
  /// This constructor also initializes DefaultNS.
  SequenceSet(void): DefaultNS()
   {
   Separator="\n";
   }

  /// \brief The constructor which takes window_size as argument.
  /// This constructor also initializes DefaultNS.
  SequenceSet(typename sequence_type::window_type ws) ///< NucleotideSequence::window_size, will the same for all
                                                      ///< sequences that are read in.
   : DefaultNS(ws)
   {
   Separator="\n";
   }

  /// \brief Searches for sequences that start with val
  /// \param start search from here until end of container
  /// \param val   search for sequences beginning with this value
  inline typename container_type::iterator find_starting(typename container_type::iterator &start,
                                                         const sequence_type &val)
  {
     if(val.empty()) return container_type::end();

     for(typename container_type::iterator i = start;i != container_type::end();i++) {
       if(std::equal(val.begin(),val.end(),i->begin())) {
         return i;
       }
     }

     return container_type::end();
  }


  /// \brief Insertion of a NucleotideSequence into SequenceSet
  /// \attention This uses function set::insert which may not exist of other container classes.
  inline SequenceSet& operator<<(sequence_type& ns)
    {
    this->insert(ns);
    return *this;
    }

  /// \brief Extractor which prints the whole sequence.
  inline friend std::ostream& operator<<(std::ostream& out, const SequenceSet& seqset)
    {
    std::copy(seqset.begin(),seqset.end(),std::ostream_iterator<sequence_type>(out,seqset.Separator.c_str()));
    return out;
    }

}; //class

  template<>
  /// Specialization for std::deque and NucleotideSequence since std::deque has function insert with different syntax.
  inline SequenceSet<NucleotideSequence<>,std::deque<NucleotideSequence<> > >& SequenceSet<NucleotideSequence<>,std::deque<NucleotideSequence<> > >::operator<<(NucleotideSequence<>& ns)
    {
    insert(end(),ns);
    return *this;
    }

  template<>
  /// Specialization for std::multimap<NucleotideSequence<>,int>
  inline SequenceSet<NucleotideSequence<>,std::multimap<NucleotideSequence<>,int > >& SequenceSet<NucleotideSequence<>,std::multimap<NucleotideSequence<>,int > >::operator<<(NucleotideSequence<>& ns)
    {
    insert(sequenceset_type::value_type(ns,0));
    return *this;
    }

  inline std::ostream& operator<<(std::ostream& out, const std::multimap<NucleotideSequence<>,int >::value_type& sp)
    {
    out << (std::string)sp.first << " " << sp.second;
    return out;
    }

}; // namespace

#endif
