// $Id: Actions.h 1112 2020-05-03 22:14:17Z ge $
/// \file TransferMatrix.h
/// \brief contains the Hamiltonian function class
///
/// $Revision: 1112 $
/// \author Gerald Weber <gweberbh@gmail.com>

#ifndef GBC_ACTIONS_H
#define GBC_ACTIONS_H "$Id: Actions.h 1112 2020-05-03 22:14:17Z ge $"

#include <valarray>
#include "HeterogenousTM.h"

namespace gbc
  {
  /// \brief Does nothing, just a placeholder.
  class ApplyNothing
    {
    public:
      template<class _TF>
      static void apply(_TF& tf,const Range<double>& T) {}
    };

  /// \brief Changes the temperature from the values of a Range.
  class ApplyTemperature
    {
    public:
      template<class _TF>
      static void apply(_TF& tf,const Range<double>& T) 
        {tf.temperature(T);}
    };

  /// \brief Changes the temperature from the values of a Range.
  class ApplyPosition
    {
    public:
      template<class _TF>
      static void apply(_TF& tf,const Range<double>& y) 
        {tf.Position=y;}
    };

 /// \brief Changes the upper integration limit from the values of a Range.
  class ApplyUpperLimit
    {
    public:
      template<class _TF>
      static void apply(_TF& tf,const Range<double>& R) 
        {tf.limits(tf.Lower_limit,R,tf.Integration_steps);}
    };

 /// \brief Changes the upper integration limit from the values of a Range.
  class ApplyLimit
    {
    public:
      template<class _TF>
      static void apply(_TF& tf,const Range<double>& integration) 
        {tf.limits(integration.begin(),integration.end(),integration.steps());}
    };

 /// \brief Placeholder.
  class ActionDoNothing
    {
    public:
      template<class _TF>
      inline std::valarray<typename _TF::value_type> action(_TF& tf,size_t eigen_max)
        {
        return std::valarray<typename _TF::value_type>();
        }
    };

 /// \brief Placeholder.
  class ActionDeltaIndex
    {
    public:
      template<class _TF>
      inline std::valarray<typename _TF::value_type> action(_TF& tf,size_t eigen_max)
        {
        return std::valarray<typename _TF::value_type>();
        }
    };

 /// \brief Calculates the Eigenvalue only.
  class ActionEigenvalue
    {
    public:
      template<class _TF>
      std::valarray<typename _TF::value_type> action(_TF& tf,size_t eigen_max)
        {
        tf.calculate_eigenvalue_only(true);
        tf.diagonalize();
        return std::valarray<typename _TF::value_type>(tf.eigen_engine.Eigenvalue.Array[std::slice(0,eigen_max,1)]);
        }
    };

 /// \brief Calculates the Eigenvector only.
  class ActionEigenvector
    {
    public:
      template<class _TF>
      std::valarray<typename _TF::value_type> action(_TF& tf,size_t n)
        {
        tf.calculate_eigenvalue_only(false);
        tf.diagonalize();
        return tf.eigenvector(n);
        }
       template<class _TF>
       std::valarray<typename _TF::value_type> index(_TF& tf) 
         {return tf.Integration_space;}
     };

// \brief Calculates the Eigenvector only.
  class ActionEigenfunction
    {
    public:
      template<class _TF>
      std::valarray<typename _TF::value_type> action(_TF& tf,size_t n)
        {
        tf.calculate_eigenvalue_only(false);
        tf.diagonalize();
        typename _TF::value_type ef=tf.eigenfunction(tf.Position,n);
        return std::valarray<typename _TF::value_type>(ef,1);
        }
    };

  /// \brief Calculates the \f$\angle n| y |n\angle$\f..
  class ActionAverageYnn
    {
    public:
      template<class _TF>
      std::valarray<typename _TF::value_type> action(_TF& tf,size_t n)
        {
        tf.calculate_eigenvalue_only(false);
        tf.diagonalize();
        return std::valarray<typename _TF::value_type>(tf.average_y(n,n),1);
        }
    };

  /// \brief Calculates the \f$\angle y \angle$\f..
  class ActionAverageY
    {
    public:
      size_t number_of_basepairs;
      template<class _TF>
      std::valarray<typename _TF::value_type> action(_TF& tf,size_t N)
        {
        tf.calculate_eigenvalue_only(false);
        tf.diagonalize();
        return std::valarray<typename _TF::value_type>(tf.average_y(N),1);
        }
      template<class _TF>
      std::valarray<typename _TF::value_type> index(_TF& tf) 
       {
       std::valarray<typename _TF::value_type> in(number_of_basepairs);
       for (size_t i=0; i < number_of_basepairs; ++i) in[i]=i;
       return in;
       }
    };


  /// \brief Calculates the \f$\angle y \angle$\f..
  class ActionOpenAverageY
    {
    public:
      size_t number_of_basepairs;
      template<class _TF>
      std::valarray<typename _TF::value_type> action(_TF& tf,size_t N)
        {
        number_of_basepairs=N;
        tf.calculate_eigenvalue_only(false);
        tf.diagonalize();
        return tf.open_average_y(N);
        }
      template<class _TF>
      std::valarray<typename _TF::value_type> index(_TF& tf) 
       {
       std::valarray<typename _TF::value_type> in(number_of_basepairs);
       for (size_t i=0; i < number_of_basepairs; ++i) in[i]=i;
       return in;
       }
    };
    
  /// \brief Calculates the Zy and Fy
  class ActionZyFy
    {
    public:
      size_t number_of_basepairs;
      template<class _TF>
      std::valarray<typename _TF::value_type> action(_TF& tf,size_t N) //size_t N is ignored
        {
        std::valarray<typename _TF::value_type> in(2);
        tf.calculate_eigenvalue_only(false);
        tf.calculate_eigensystem_and_matrices();
        in[0] = tf.partition_function();
        in[1] = tf.helmoltz_energy();
        return in;
        }
     };

     
       /// \brief Calculates the Zy and Fy
  class ActionZyNoRecalculate
    {
    public:
      size_t number_of_basepairs;
      template<class _TF>
      inline typename _TF::value_type value(_TF& tf,const size_t& N)
        {
        tf.retrieve_or_calculate_matrices();
        return tf.partition_function();//implicitly calculates Fy also
        }
     };

  /// \brief Expands Zy in terms of Lambda
  class ActionExpandZy
    {
    public:
      size_t number_of_basepairs;
      template<class _TF>
      std::valarray<typename _TF::value_type> action(_TF& tf,size_t N) //size_t N is ignored
        {
        std::valarray<typename _TF::value_type> in(tf.pSI->Exact_neighbours.size());
        tf.calculate_eigenvalue_only(false);
        tf.diagonalize();
        in = tf.expand_partition_function();
        return in;
        }
     };


  class ActionIndex
    {
    public:
      static std::string name(void) {return std::string("ActionIndex");}
      template<class _TF>
      std::valarray<typename _TF::value_type> action(_TF& tf,size_t N)
        {
        tf.retrieve_or_calculate_matrices();
        return std::valarray<typename _TF::value_type>(tf.melting_index(),1);
        }
      template<class _TF>
      inline typename _TF::value_type value(_TF& tf,const size_t& N)
        {
        tf.retrieve_or_calculate_matrices();
        return tf.melting_index();
        }
    };

  class ActionApproxIndex
    {
    public:
      static std::string name(void) {return std::string("ActionApproxIndex");}
      template<class _TF>
      std::valarray<typename _TF::value_type> action(_TF& tf,size_t N)
        {
        tf.eigenvalue_cutoff(1);
        tf.retrieve_or_calculate_matrices();
        return std::valarray<typename _TF::value_type>(tf.approx_melting_index(),1);
        }
      template<class _TF>
      inline typename _TF::value_type value(_TF& tf,const size_t& N)
        {
        tf.eigenvalue_cutoff(1);
        tf.retrieve_or_calculate_matrices();
        return tf.approx_melting_index();
        }
    };

  class ActionNoMatrixApproxIndex
    {
    public:
      template<class _TF>
      inline std::valarray<typename _TF::value_type> action(_TF& tf,size_t N)
        {
        return std::valarray<typename _TF::value_type>(tf.approx_melting_index(),1);
        }
      template<class _TF>
      inline typename _TF::value_type value(_TF& tf,const size_t& N)
        {
        return tf.approx_melting_index();
        }
    };

  class ActionLoopApproxIndex
    {
    public:
      template<class _TF>
      inline std::valarray<typename _TF::value_type> action(_TF& tf,size_t N)
        {
        return std::valarray<typename _TF::value_type>(tf.approx_melting_index_loop(),1);
        }
      template<class _TF>
      inline typename _TF::value_type value(_TF& tf,const size_t& N)
        {
        return tf.approx_melting_index_loop();
        }
    };
  class ActionCGcontent
    {
    public:
      static std::string name(void) {return std::string("ActionCGcontent");}
      template<class _TF>
      std::valarray<typename _TF::value_type> action(_TF& tf,size_t N)
        {
        return std::valarray<typename _TF::value_type>(tf.cg_content(),1);
        }
      template<class _TF>
      inline typename _TF::value_type value(_TF& tf,const size_t& N)
        {
        return tf.cg_content();
        }
    };


  template<class _Action, class _Apply=ApplyNothing>
  class Looping
    {
    public:
    template<class _TM>
    /// \brief Loops over a range and performs an action for each loop and prints the result.
    static void loop(_TM& tm, Range<double> r, std::ostream& out=std::cout, size_t eigen_max=1)
      {
      _Action action;
      for(; !r.finished(); ++r)
        {
        _Apply::apply(tm,r);
        typename _TM::vector_type result=action.action(tm,eigen_max);
        out << r.current() << " " << result << std::endl;
        }
      }

    /// \brief Loops over a range and performs an action for each loop and prints the result.
    // old name static void result_vector(_TM tm, size_t eigen_max=1,std::ostream& out=std::cout)
    template<class _TM>
    static void print_result(_TM tm, size_t eigen_max=1,std::ostream& out=std::cout)
      {
      _Action action;
      typename _TM::vector_type result= action.action(tm,eigen_max), index=action.index(tm);
      size_t n;
      for (n=0; n < result.size(); n++)
        {
        out << index[n] << " " << result[n] << std::endl;
        }
      }

    /// \brief Loops over a range and performs an action for each loop and prints the result.
    template<class _TM>
    static void result_vector(typename _TM::vector_type& res,_TM& tm, size_t eigen_max=1)
      {
      _Action action;
      res=action.action(tm,eigen_max);
      }

    template<class _TM>
    inline static void result_value(typename _TM::value_type& res,_TM& tm, size_t eigen_max=1)
      {
      _Action action;
      res=action.value(tm,eigen_max);
      }
    };


/*  template<class _TM>
  class Actions
    {
    public:
      std::ofstream outfile;
    };
*/
};
#endif
