// $Id: ErrorCodes.h 1369 2024-11-29 14:20:00Z ge $
/// \file ErrorCodes.h
/// \brief contains predefined error codes
///
/// $Revision: 1369 $
/// \author Gerald Weber <gweberbh@gmail.com>
#ifndef ERRORCODES_H
#define ERRORCODES_H "$Id: ErrorCodes.h 1369 2024-11-29 14:20:00Z ge $"

#include <string>
#include <map>


namespace gbc {
class DebugControl
  {
  public:
    typedef std::map<std::string,bool> debug_map_type;
    
  private:
    static debug_map_type DebugMap;
    static bool EmptyDebugMap;
    static bool DebugAll;
    static bool RegexCodes;
    
  public:
    
    DebugControl(void);
    
    static void enable(std::string code);
    
    static void disable(std::string code);

    static void enable_all(void);
    
    static void disable_all(void);

    static bool debug(std::string code);
    
  };
}

//requires <iomanip> when used
#define MAXPRECSION std::setprecision(std::numeric_limits<long double>::digits10 + 1)

#define CERR std::cerr 

#define COUT std::cout 

#define CERR_ERROR(CODE) \
std::cerr << ":" << CODE << ":" << __FILE__ << "@" << __LINE__ << ": Error "

#define CERR_IERROR(CODE) \
std::cerr << ":" << CODE << ":" << __FILE__ << "@" << __LINE__ << ": Internal Error "

#define CERR_WARN(CODE) \
std::cerr << ":" << CODE << ": Warning "

#define BOOL_DDEBUG(CODE) \
(Debug or DebugControl::debug(CODE))

#define BOOL_DEBUG(CODE) \
DebugControl::debug(CODE)

#define CERR_DEBUG_CODE(CODE) \
if (DebugControl::debug(CODE)) std::cerr << ":" << CODE << ":" 

#define CERR_DEBUG(CODE) \
if (DebugControl::debug(CODE)) std::cerr << ":" << CODE << ":" << __FILE__ << "@" << __LINE__ << ":"

  
//For classes that have their own Debug flag
#define CERR_DDEBUG(CODE) \
if (Debug or DebugControl::debug(CODE)) std::cerr << ":" << CODE << ":" << __FILE__ << "@" << __LINE__ << ":"

#define CERR_DDEBUG_NH(CODE) \
if (Debug or DebugControl::debug(CODE)) std::cerr 

#define IFDDEBUG(CODE) \
if (Debug or DebugControl::debug(CODE))
  
#define STRING_CODE_PRF(CODE) \
(std::string(":") + std::string(CODE) + std::string(":"))
  
#define COUT_INFO(CODE) \
std::cout << ":" << CODE << ":"

#define CERR_TERM std::cerr << "Program terminating" << std::endl; exit(1);

//Problems with numerical routines start with 100
#define ERROR_REGRESSION   100 //(Debug)Linear GSL regression returned NaN values

//Problems with specific calculations start with 200
#define ERROR_MELTINGINDEX 200 //(removed)Melting index returned zero

//Problems with base pair or NN interpretation start with 300
#define ERROR_VOIDEXACTNEIGHBOURS 300 //Void pSI->Exact_neighbours

//Problems with options start with 400
#define ERROR_WRONG_PG_VALUE 400 //Unrecognized -pg value
#define ERROR_WRONG_PM_VALUE 401 //Unrecognized -pm value

//modules/ErrorCodes.cpp
#define INFOEDF   "INFOEDF"  //enabling debug flag
#define INFODFREG "INFODFREG" //debug flag detected as regular expression
#define INFOAEDF  "INFOAEDF" //enabling all debug flags
#define INFODSDF  "INFODSDF" //disabling all debug flags
#define INFORDF   "INFORDF"  //removing debug flag
#define INFODFNF  "INFODFNF" //debug flag not found

//include/NucleotideDictionary.h
#define IERRNNMIF "IERRNNMIF" //no NucleotideCode::Math implementation for NucleotideDictionary<int,char>
#define IERRRDUE  "IERRRDUE" //read_dictionaryUnknown exception
#define IERRSIBF  "IERRSIBF" //std::ios_base::failure
#define WRIIWR    "WRIIWR"   //Warning: replacing internal index " << smb << " with representation
#define WRCF      "WRCF"     //replacing complementary for
#define WCRNF     "WCRNF"    //complementary representation for \"" << errcnv <<  "\" not found
#define DNUD_NISFF    "DNUD_NISFF"   //no inverse symbol found for
#define DNUD_SMBCNV   "DNUD_SMBCNV"  //"smb=" << smb << " cnv="
#define DNUD_ADDNEW   "DNUD_ADDNEW"  //adding new representation

//include/Nucleotide.h
#define ERRUST    "ERRUST"    //unknown sugar type
#define ERRCNIB   "ERRCNIB"   //could not interpret BasePair
#define DNUC_STRINT "DNUC_STRINT"  //string_interpretation

//include/NucleotideSequence.h
#define DNUS_ASSIGN  "DNUS_ASSIGN"  //assign function
#define DNUS_RESULT  "DNUS_RESULT"  //show (*result)[0].str()

//include/OutStream.h
#define IERROSNO  "IERROSNO"   //Output stream not open
#define DOUS_BUILDFL  "DOUS_BUILDFL"   //build_filename
#define DOUS_FLDEL "DOUS_FLDEL"  //delete file
#define DOUS_FLCLS  "DOUS_FLCLS" //closing file, note: explicit closing of files may not show always, that is, you should not be concernced if you don't see a "closing" after seeing a "opening"
#define DOUS_FLOPN  "DOUS_FLOPN" //opening file
#define DOUS_FLNEX  "DOUS_FLNEX"   //file does not exists

//include/ReferenceMap.h
#define WPFBPNNF "WPFBPNNF" //Parameters for " << bpn << " not found"

//include/ReferenceSet.h
#define DRES_BPNSR "DRES_BPNSR"        //bpn symmetry action results before/after
#define DRES_NEWBPN "DRES_NEWBPN"      //new base pair

//include/StrandPairSequence.h
#define WNRFSPF "WNRFSP" //Warning no reference for " << sp << " found"

//base/options/Options.cpp
#define ERRAGNAFO "ERRAGNAFO" //"Error: argument given \"" << readin << "\" not acceptable for option
#define ERRMOCYP  "ERRMOCYP" //missing options, check your parameters or see documentation
#define ERRNAPAE  "ERRNAPAE" //no arguments provided after equal sign (=)  for option
#define ERRNAPSLA "ERRNAPSLA" //"No arguments provided, showing list of available arguments and exiting
#define IERRPWNAD "IERRPWNAD" //Option with name " << arg <<  " already defined, ending program
#define WROPTASSYN "WROPTASSYN" //Warning: reading \"-" << opt << "\" as \"-" << syn_it->second
#define INFOREDENV "INFOREDENV" //reading options from environment variable
#define DOPT_SHEQ  "DOPT_SHEQ" //i=" << i << " arg=

//base/options/Range.h
#define ERRUIEAAR "ERRUIEAAR" //unable to interpret \"" << expr << "\" as a range"
#define WRCPPMTL  "WRCPPMTL"  //Re-centralize problems, pm too large
#define WRCPCSTL  "WRCPCSTL"  //Re-centralize problems, Current_step too large
#define DRAN_C2V  "DRAN_C2V"  //Shows the range during conversion from Range to std::valarray
#define DRAN_FIN  "DRAN_FIN"  //shows values for end_condition

//base/sequencestructure/SequenceInfo.h
#define ERRCBRMT  "ERRCBRMT" //"Central base pair of regex matched trimer \"" << tmp << "\" is " << trimer[1].formatted_string_no_version() << " which does not match rule " << re.first << ", check your regular expression for mistahces"
#define WF1F2F3Z "WF1F2F3Z" //factors f1,f2 and f3 are all zero
#define WIATAM    "WIATAM"     //In additon to /" << first_match << " regex /" << re.first << "/ also matches "

//base/sequencestructure/SequenceDataset.h
#define ERRSCSDNE "ERRSCSDNE"   //Error: Salt correction scheme " << scheme << " does not exists
#define ERRESBER  "ERRESBER"   //Error: ExclusivePNeighbours_set.size()+NPNeighbours_set.size() should be equal to Reduced_neighbours_set.size()
#define ERRFXDNE "ERRFXDNE"    //"File " << xmlfile << " does not exist"
#define ERRFRDNE "ERRFRDNE"    //File " << rulefile << " does not exist
#define ERRCNRR  "ERRCNRR"     //Could not read " << rulefile
#define ERRCNIRL "ERRCNIRL"    //Could not interpret rule line [" << line << "] check for formatting mistakes
#define ERRPMT   "ERRPMT"      //Possibly malformed trimer
#define ERRUTPL  "ERRUTPL"     //Error unable to parse line
#define ERRFDFXNR "ERRFDFXNR"  //Format of data file " << xmlfile.c_str() << " not recognized
#define ERRNOSEQR "ERRNOSEQR"  //No sequences were read
#define WSCIZ "WSCIZ"          //warning: species concentration is zero (Ct=0)
#define WDHMSC    "WDHMSC"     //dataset " << Dataset_filename << " has multiple salt concentrations
#define WCTITPA   "WCTITPA"    //Check trimer \"" << el << "\", its reverse \"" << le << "\" precedes alphabetically, therefore it may never be matched to any existing trimer"
#define WTFRAEFAR "WTFRAEFAR"  //"Trimer \"" << el << "\" for rule " << bp << " already exists for a another rule " << fnd->second << ", check your trimer rules for duplicates"
#define WCBPTWDNMR "WCBPTWDNMR" //Central base pair of trimer which does not match rule
#define WSHNCURC  "WSHNCURC"   //Sequence " << seq << "/" << comp << "has no Ct, using the reference Ct=" <<  Species_concentration
#define WSHNCTCMF "WSHNCTCMF"  //" Sequence " << seq << "/" << comp << "has no Ct, Tm calculation may fail"
#define DSED_EXSPLIT  "DSED_EXSPLIT"     //extracted split
#define DSED_TMDG "DSED_TMDG" //show inserted DeltaG and Tm
#define DSED_TMDGREG "DSED_TMDGREG" //show regression parameters of Tm x DeltaG
#define DSED_TMDGVEC "DSED_TMDGVEC" //show regression vectors
#define DSED_RXML "DSED_RXML" //read_xml_line
#define DSED_SISCMC "DSED_SISCMC" //Sequence is self-complementary, multiplying Ct= by
#define DSED_PRTFLG "DSED_PRTFLG" //PrintFlags
#define INFOSRSC  "INFOSRSC"   //Setting reference species concentration to
#define INFODHMSC "INFODHMSC"  //dataset " << Dataset_filename << " has multiple species concentrations
#define INFOSBSCT "INFOSBSCT"  //Setting base salt concentration to " <<  Salt_concentration << " from sequence
#define INFOTRRF  "INFOTRRF"   //trimer rules read from file
#define INFOTWMTL "INFOTWMTL"  //Trimer equivalence map too long, output suppressed
#define INFOTRMAT "INFOTRMAT"  //Trimer match
#define INFORDFL  "INFORDFL"   //Reading data file
#define INFORSCW  "INFORSCW"   //Replacing sequence Ct

//base/parameters/ParameterMap.h
#define ERRNPF "ERRNPF" //"no specific or generic parameter found for "
#define ERRCNRF "ERRCNRF" //Could not read " << fname
#define WUBPG "WUBPG" //Warning: using BP generic
#define WUNNG "WUNNG" //Warning: using NN generic
#define WPIDNF "WPIDNF" //Warning set_parameters(): parameter " << id << " not found
#define WROVNV "WROVNV" //Warning:replacing old value= new value=
#define DPMA_UGF  "DPMA_UGF"  //Debug: using " << generic << " for
#define INFORPF "INFORPF" //Reading parameters from

//base/parameters/ParameterMap.h
#define ERRCHMP "ERRCHMP" //"cannot handle multiple prefixes for "
#define ERRTJDPT "ERRTJDPT"

//base/parameters/ParameterSearch.h
#define ERRUNMINEN "ERRUNMINEN" //Unknown minimization engine
#define ERRVARNOTU "ERRVARNOTU" //Variable " << mit.first << " is not part of parameter used
#define ERRCLNOTP  "ERRCLNOTP"  //could not parse
#define INFOVARPAR "INFOVARPAR" //"Variable parameter = " << variable;
#define INFOIVRMR  "INFOIVRMR"  //Initial variables randomised for minimization round (mr)
#define INFONBOPR  "INFONBOPR"  //"New best set of parameters at round
#define INFOMNRLMR "INFOMNRLMR" //Maximum number of randomized local minimizations reached
#define INFOLTADR  "INFOLTADR"  //Limit for total average difference reached
#define INFOLTSDDR "INFOLTSDDR" //Limit for total standard deviation difference reached
#define INFOCPFDEC "INFOCPFDEC" //Change in proceed file detected, ending calculation
#define INFODBTLTR "INFODBTLTR" //Difference between the last two rounds
#define INFOLMSAR  "INFOLMSAR"  //Local minimization stopped at round
#define INFOMM0D   "INFOMM0D"   //-mm=0 detected (meaning inf) maximium number of minimizations set to
#define DPAS_ASSIGNVAR "DPAS_ASSIGNVAR" //assign_variables
#define DPAS_EVCALPM   "DPAS_EVCALPM"   //  Evaluation.Calculation.parameter_map.print_parameters_and_stats
#define DPAS_EVALRES   "DPAS_EVALRES"   // evalresult
#define DPAS_VMAPRAND  "DPAS_VMAPRAND"  //(variable_map_randomise)
#define DPAS_VMAPREA   "DPAS_VMAPREA"   //variable_map_reassign
#define DPAS_AINTTO "DPAS_AINTTO" //Assigning interval to
#define DPAS_AVRI   "DPAS_AVRI"   //assigned variable with increment
#define DPAS_SHSVDR "DPAS_SHSVDR" //show SVD x result

//math/eigen/EigenSystem.h
#define ERRCDINAN "ERRCDINAN" //cannot diagonalize infinite or NaN vec
#define ERRPRWDIA "ERRPRWDIA" //problems with diagonalization
#define ERRTHARGI "ERRTHARGI" //the" <<  retval << "-th argument had an illegal value
#define ERRQRALFA "ERRQRALFA" //QR algorithm failed to compute all the eigenvalues

//math/legendre/Legendre.h
#define DLEG_LOADF "DLEG_LOADF" //Loading [" << filename()
#define DLEG_FNF   "DLEG_FNF" //File [" << filename().c_str() << "] not found
#define DLEG_SAVFI "DLEG_SAVFI"  //Saving [" << filename().c_str() << "]"

//math/minimization/LinearMinimization.h
#define ERRSVDECOMP "ERRSVDECOMP" //Failed SVD decomposition
#define ERRSVSOLVE  "ERRSVSOLVE"  //Failed SVD solve
#define INFOSVDSRI  "INFOSVDSRI"  //SVD sucessful, S rank is
#define DLMI_PRNTS  "DLMI_PRNTS"  //prints the S matrix

//math/matrix/valmatrix.h
#define ERRTAVCS   "ERRTAVCS" //Trying to assign vector of size
#define ERRSUBNPOS "ERRSUBNPOS" //Submatrix of (first_row,first_column,last_row,last_column)  from a matrix of size Rows x  Columns is not possible.
#define ERRMATDNA  "ERRMATDNA" //Matrices of dimensions can not be added
#define ERRMATDNS  "ERRMATDNS" //Matrices of dimensions can not be subtracted
#define ERRMATDNM  "ERRMATDNM" //Matrices of dimensions can not be multiplied
#define ERRMFNCOMP "ERRMFNCOMP" //Matrix file seems not to be compatible with
#define ERRMFSST   "ERRMFSST" //Matrix in file is of size which smaller than

//math/minimization/Minimization.h
#define WMINPCIP   "WMINPCIP" //Minimization warning! poor choice of initial points
#define DMIN_BSTN "DMIN_BSTN" //best()[n]

//math/minimization/GSLMinimization.h
#define ERRGSLERR "ERRGSLERR" //gsl_strerror

//math/regression/Regression.h
#define DREG_FSRCRNAN "DREG_FSRCRNAN" //regression failed, some regression coefficients returned NaN

//projects/align/dynamic/NeedlemanWunsch.h
#define IERRWRECD  "IERRWRECD"  //wrong RecurrenceDirection, requires program debug
#define DNEE_BTPOS "DNEE_BTPOS" //print backtrace start position and value of AlignmentMatrix
#define DNEE_SHBTP "DNEE_SHBTP" //print backtrace positions
#define DNEE_PMADQ "DNEE_PMADQ" //print MainAlignment deque immediately after backtrace
#define DNEE_PTGDQ "DNEE_PTGDQ" //print TargetAlignment deque immediately after backtrace
#define DNEE_PAMBK "DNEE_PAMBK" //print alignment matrix with backtrace
#define DNEE_PAMWF "DNEE_PAMWF" //print alignment matrix while it is filling up
#define DNEE_PFAM  "DNEE_PFAM"  //print alignment matrix after the filling process is finished
#define DNEE_SHLTA "DNEE_SHLTA" //print last target alignment
#define DNEE_SHLMA "DNEE_SHLMA" //print last main alignment

//projects/align/dynamic/Nussinov.h
#define NUSS_PFAM  "NUSS_PFAM"  //print alignment matrix after the filling process is finished

//projects/align/dynamic/SmithWaterman.h
#define DSMW_CHAL  "DSMW_CHAL"  //print change in highest alignment
#define DSMW_SHHD  "DSMW_SHHD"  //print highest diagonal

//projects/align/dynamic/align_primer.cpp
#define ERRINVBPLEN "ERRINVBPLEN" //invalid BP length
#define INFOADDBP "INFOADDBP" //add acceptable BP

//projects/align/dynamic/AlignSequence.h
#define ERRRFNOP   "ERRRFNOP"   //report file not open
#define WSSMEL     "WSSMEL"   //The selected score may exceed limits
#define DALS_PMDIRD "DALS_PMDIRD" //print Main in dir=D

///projects/base/sequencestructure/SequenceInfo.h
#define ERRSCNWC "ERRSCNWC" //salt correction nakano99 wrong concentrations
#define ERRTSCZ "ERRTSCZ" //target salt concentration is zero
#define ERRCCSC "ERRCCSC" //Cannot calculate salt correction with zero enthaply for santalucia98
#define DSQI_PRTFLG "DSQI_PRTFLG" //shows PrintFlags
#define DSQI_CLABWD "DSQI_CLABWD" //calculate_absorption_width
#define DSQI_INSSEQ "DSQI_INSSEQ" //shows insert_sequence

///FEM /projects/thermodynamics/gibbs/ModelDataSet (old FreeEnergyModel.h)
#define DFEM_SHSVDMAT "DFEM_SHSVDMAT" //show SVD matrix
#define DFEM_SHCHI2 "DFEM_SHCHI2" //show chi2 for predict_all_adjusting

//INN projects/thermodynamics/gibbs/IndividualNearestNeighbours.h
#define ERRNOVAR  "ERRNOVAR" //no such variable
#define WINNSCIZ "WINNSCIZ" //species concentration is zero
#define WNSEEV "WNSEEV" //no specific Enthalpy or Entropy values, falling back to generic values
#define DINN_HSG "DINN_HSG" //dArU_dCrG DeltaH = -9.5 DeltaS = -26 DeltaG = -1.4361
#define DINN_PTHSG "DINN_PTHSG" //Partial total (before initiation) DeltaH = -160.7 DeltaS = -439.3 Delta G(37C) = -24.4511
#define DINN_FPN "DINN_FPN" //formatted_parameter_name
#define DINN_SIED "DINN_SIED" //sequence Sequence_info->Exact_duplex
#define DINN_TMATENTH "DINN_TMATENTH" //Terminal_AT_enthalpy
#define DINN_TMATENTR "DINN_TMATENTR" //Terminal_AT_entropy
#define DINN_TMATDLTG "DINN_TMATDLTG" //Terminal_AT_DeltaG
#define DINN_TMCGENTH "DINN_TMCGENTH" //Terminal_CG_enthalpy
#define DINN_TMCGENTR "DINN_TMCGENTR" //Terminal_CG_entropy
#define DINN_TMCGDLTG "DINN_TMCGDLTG" //Terminal_CG_DeltaG
#define DINN_TDARUDTRAONLYENTH "DINN_TDARUDTRAONLYENTH" //Terminal_dArU_dTrA_only_enthalpy
#define DINN_TDARUDTRAONLYENTR "DINN_TDARUDTRAONLYENTR" //Terminal_dArU_dTrA_only_entropy
#define DINN_TDARUDTRAONLYDLTG "DINN_TDARUDTRAONLYDLTG" //Terminal_dArU_dTrA_only_DeltaG
#define DINN_TDCRGDGRCENTH "DINN_TDCRGDGRCENTH" //Terminal_dCrG_dGrC_enthalpy
#define DINN_TDCRGDGRCENTR "DINN_TDCRGDGRCENTR" //Terminal_dCrG_dGrC_entropy
#define DINN_TDCRGDGRCDLTG "DINN_TDCRGDGRCDLTG" //Terminal_dCrG_dGrC_DeltaG
#define DINN_INENTH "DINN_INENTH" //Initiation_enthalpy
#define DINN_INENTR "DINN_INENTR" //Initiation_entropy
#define DINN_INDLTG "DINN_INDLTG" //Initiation_DeltaG
#define DINN_AEET   "DINN_AEET"   //Allocating enthalpy/entropy tables
#define DINN_SYCORRENTR "DINN_SYCORRENTR" //Symmetry correction entropy
#define DINN_SYCORRENTH "DINN_SYCORRENTH" //Symmetry correction enthalpy
#define DINN_TDHDSDG "DINN_TDHDSDG" //Total DH = -149.7 Total DS = -413.8 Total DG = -21.3599
#define DINN_INATENTH "DINN_INATENTH" //Initiation AT enthalpy
#define DINN_INATENTR "DINN_INATENTR" //Initiation AT entropy
#define DINN_INCGENTH "DINN_INCGENTH" //Initiation CG enthalpy
#define DINN_INCGENTR "DINN_INCGENTR" //Initiation CG entropy
#define DINN_INCGDELG "DINN_INCGDELG" //Initiation CG DeltaG
#define DINN_T5AT3ENTH "DINN_T5AT3ENTH" //Terminal_5TA3_enthalpy = " << Sequence_info->Terminal_5TA3 << " X " << Terminal_5TA3_enthalpy
#define DINN_TAUENTH  "DINN_TAUENTH" //Terminal_AU_enthalpy
#define DINN_TAUENTR  "DINN_TAUENTR" //Terminal_AU_entropy
#define INFOSPSCT "INFOSPSCT" //setting parameter salt concentration to

//projects/thermodynamics/gibbs/GibbsMinimization.h
#define DGBM_CHI2 "DGBM_CHI2" //show chi2  

//projects/thermodynamics/gibbs/vargibbs.cpp
#define INFOPMERTN "INFOPMERTN" //print merit function name 

//projects/thermodynamics/transfermatrix/ModelPotentials.h
#define DMODPOT_MORPREF "DMODPOT_MORPREF" //show Morse prefix

//projects/thermodynamics/transfermatrix/HeterogenousTM.h
#define ERRNMF "ERRNMF"    //No matrix for [" << nm << "]
#define ERRVENI "ERRVENI"  //Void pSI->Exact_neighbours in
#define DHTM_NBDFM  "DHTM_NBDFM"   //New base directory for matrices
#define DHTM_ANMFNN "DHTM_ANMFNN"  //Adding new matrix for NN=
#define DHTM_NMYFNN "DHTM_NMYFNN"  //No matrix yet for NN=
#define DHTM_NMCFNN "DHTM_NMCFNN"  //New matrix created for NN=
#define DHTM_PC     "DHTM_PC"      //Parameters_change
#define DHTM_READFN "DHTM_READFN"  //"Reading " << filename
#define DHTM_CREAMAP "DHTM_CREAMAP" //Created " << TM_map[it->first]
#define DHTM_RETCALM "DHTM_RETCALM" //retrieve_or_calculate_matrices
#define DHTM_RECDIAG "DHTM_RECDIAG" //Recalculate_diagonalization=" << Base->second.TM->Recalculate_diagonalization  Parameters_changed
#define DHTM_SKIPDIA "DHTM_SKIPDIA"  //skipping diagonalization of
#define DHTM_RETEIG  "DHTM_RETEIG"   //retrieve_eigensystem
#define DHTM_EXPBASE "DHTM_EXPBASE"  //expand on base
#define INFOFNFC "INFOFNFC"    //filename << " not found, calculating

//projects/thermodynamics/transfermatrix/ModelPotentials.h
#define DMOP_HCF   "DMOP_HCF"  //the calculated a, b, c, and A factors of the hump potential

//bioinf/projects/thermodynamics/transfermatrix/Potentials.h
#define IERRPNDEF "IERRPNDE" //potential not defined
#define DPOT_DFPAR "DPOT_DFPAR" // parameter defined by potential

//projects/thermodynamics/transfermatrix/RegressionMinimization.h
#define DRMI_SQRDIFF "DRMI_SQRDIFF" //Regression.dataset.Temperature.sqr_diff

//projects/thermodynamics/transfermatrix/RegressionModel.h
#define IERRFSVDVNF "IERRFSVDVNF" //function set_svd_variables not defined
#define IERRFSVDMNF "IERRFSVDMNF" //function set_svd_matrices not defined
#define IERRFASVDUNF "IERRFASVDUNF" //function adjust_svd_units not defined
#define ERRCNOF  "ERRCNOF"   //Could not open file
#define ERRNRPFSC  "ERRNRPFSC"   //predict(): No regression parameters for salt concentration
#define WLSEMPTY "WLSEMPTY"   //Length set empty
#define INFORRCF "INFORRCF"  //reading regression coefficients from
#define INFORTBC "INFORTBC"  //Reading type b coefficients for salt concentration
#define DRMO_THEQREG "DRMO_THEQREG"  //thermal_equivalence_reg
#define DRMO_SLENNEGA1 "DRMO_SLENNEGA1"  //alt=" << salt << " length=" << length << "negative a1
#define DRMO_SLENNEGA0 "DRMO_SLENNEGA0"  //salt=" << salt << " length=" << length << ", a0
#define DRMO_NEDPRS "DRMO_NEDPRS" //Not enough data points for regression salt
#define DRMO_PREDMETH "DRMO_PREDMETH" //prediction_method
#define DRMO_PREDGRP  "DRMO_PREDGRP"  //prediction_group
#define DRMO_PREDICT "DRMO_PREDICT" //prediction

//projects/thermodynamics/transfermatrix/TransferMatrix.h
#define ERRMHNDNE "ERRMHNDNE" //Error: Model " << hamiltonian_name << " does not exists
#define DTRM_DIAGNAL  "DTRM_DIAGNAL"  //Diagonalizing
#define DTRM_VECKXV   "DTRM_VECKXV"   //vector_type(K(xv,Integration_space))

//projects/thermodynamics/gibbs/vargibbs.cpp
#define INFOUSCDAT "INFOUSCDAT" //using salt concentration in dataset as target for salt correction
#define INFOASCS   "INFOASCS"   //applying salt correction scheme
#define INFOCSNPWC "INFOCSNPWC" //complementary sequence not provided, working out from complementary nucleotides
#define INFOCFSS   "INFOCFSS"   //Calculating for single sequence
#define INFOSSALTC "INFOSSALTC" //sequence salt concentration
#define INFOWNNATF "INFOWNNATF" //Writing nncheck analysis to file

#endif
