// $Id: ModelPotentials.h 1369 2024-11-29 14:20:00Z ge $
/// \file ModelPotentials.h
/// \brief contains the model Potentials function class
///
/// $Revision: 1369 $
/// \author Gerald Weber <gweberbh@gmail.com>
#ifndef GBC_MODELPOTENTIALS_H
#define GBC_MODELPOTENTIALS_H "$Id: ModelPotentials.h 1369 2024-11-29 14:20:00Z ge $"
#include "ErrorCodes.h"
#include "BasePair.h"
#include "BasePairNeighbours.h"
#include "StrandPair.h"
#include "Potentials.h"
using namespace gbc;
namespace gbc
  {
  template<class _Tp=double>
  /// \brief Morse potential
  ///
  /// \f$ V(y)=D\left(e^{-y/\lambda_a}-1\right)^2 \f$
  class Morse: public BasePotential<_Tp>
    {
    public:
      typedef BasePotential<_Tp> base_type;
      typedef typename base_type::parameter_map_type parameter_map_type;
      typedef _Tp value_type;
      typedef std::valarray<value_type> vector_type;

      BasePair<> bp;
      value_type lambda; ///< \f$ \lambda_a \f$
      value_type D;      ///< \f$ D \f$, depth of the potential

    Morse(void) : bp(), lambda(), D() 
      {
      base_type::name("morse");
      base_type::prefix(std::string());
      }

    Morse(const BasePair<>& b) : bp(b), lambda(), D() 
      {
      base_type::name("morse");
      bp.reduce_to_smallest_symmetry();
      CERR_DEBUG(DMODPOT_MORPREF) << "Morse prefix [" << (std::string)bp << "]"  << std::endl;
      base_type::prefix((std::string)bp);
      }

    inline bool same_par_values(const Morse &pot) const 
      {
      return lambda == pot.lambda and D == pot.D;
      }
      
    /// \brief Retrieve the parameter from the map
    inline virtual bool get_potential_parameters(parameter_map_type& pm)
      {
      bool modified_lambda = this->define_parameter(pm,"lambda",lambda);
      bool modified_D      = this->define_parameter(pm,"D",D);
      return modified_lambda || modified_D;
      }

    inline bool parameters(const Morse<value_type>& ms)
      {
      bool modified=(lambda != ms.lambda) || (D != ms.D);
      lambda=ms.lambda; 
      D=ms.D;
      return modified;
      }

    inline virtual value_type evaluate_xy(value_type x, value_type y=value_type()) const
      {
      return D*pow(exp(-x/lambda)-1.0,2);
      }
    };

  template<class _Tp=double>
  /// \brief The solvent potential.
  ///
  /// \f$ V_s(y)=-f_sD\left[\tanh(y/\lambda_s)+g_s\right] \f$
  /// \attention This also uses parameters from the Morse Potential
  class Solvent: public BasePotential<_Tp>
    {
    public:
      typedef BasePotential<_Tp> base_type;
      typedef typename base_type::parameter_map_type parameter_map_type;
      typedef _Tp value_type;
      typedef std::valarray<value_type> vector_type;

      BasePair<> bp;
      Morse<value_type> morse; ///< The Morse potential, for use of \f$ D \f$
      value_type f_s;          ///< \f$ f_s \f$
      value_type lambda;       ///< \f$ \lambda_s \f$ 
      value_type eq_sol;       ///< equilibrium distance
      value_type sign_sol;     ///< the sign \f$ g_s \f$ 

    Solvent(void) : bp(), morse(), f_s(), lambda(), eq_sol(), sign_sol() 
      {
      base_type::name("solvent");
      base_type::prefix(std::string());
      }

    Solvent(const BasePair<>& b) : bp(b), morse(b), f_s(), lambda(), eq_sol(), sign_sol() 
      {
      base_type::name("solvent");
      bp.reduce_to_smallest_symmetry();
      base_type::prefix((std::string)bp);
      }

    inline bool same_par_values(const Solvent &pot) const 
      {
      return f_s == pot.f_s and lambda == pot.lambda and eq_sol == pot.eq_sol and sign_sol == pot.sign_sol
             and morse.same_par_values(pot.morse);
      }
      
    inline virtual bool get_potential_parameters(parameter_map_type& pm)
      {
      bool modified_f_s      = this->define_parameter(pm,"f_s",f_s);
      bool modified_lambda   = this->define_parameter(pm,"lambda",lambda);
      bool modified_eq_sol   = this->define_parameter(pm,"eq_sol",eq_sol);
      bool modified_sign_sol = this->define_parameter(pm,"sign_sol",sign_sol);
      bool modified_morse    = morse.get_potential_parameters(pm);
      base_type::ParMap.insert(morse.ParMap.begin(),morse.ParMap.end());
      return modified_f_s || modified_lambda || modified_eq_sol || modified_sign_sol || modified_morse;
      }

    inline bool parameters(const Solvent<value_type>& sol)
      {
      bool modified_morse = morse.parameters(sol.morse);
      bool modified=(f_s !=sol.f_s) || (lambda != sol.lambda) || (eq_sol != sol.eq_sol) || (sign_sol != sol.sign_sol);
      f_s=sol.f_s; lambda=sol.lambda; eq_sol=sol.eq_sol; sign_sol=sol.sign_sol;
      return modified || modified_morse;
      }

    inline virtual value_type evaluate_xy(value_type x, value_type y=value_type()) const
      {
      return -f_s*morse.D*(tanh((x+eq_sol)/lambda)+sign_sol);
      }
    };

  template<class _Tp=double>
  class MorseSolvent: public CmbPotential<Morse<_Tp>,Solvent<_Tp> >
    {
    typedef  CmbPotential<Morse<_Tp>,Solvent<_Tp> > base_type;
    public:
      typedef _Tp value_type;
      MorseSolvent(void): CmbPotential<Morse<_Tp>,Solvent<_Tp> >() 
       {} 
      MorseSolvent(const BasePair<>& b): CmbPotential<Morse<_Tp>,Solvent<_Tp> >(b) 
       {} 
    };

      template<class _Tp=double>
  /// \brief A potential proportional to y^3 from peyrard09, see also martyroda18
  ///
  /// \f$ V(y)=\Theta(y)\frac{by^3}{\cosh^2\left[c(\alpha y \ln 2)\right]} \f$
  /// (03/2023) We added a p parameter such that
  /// \f$ V(y)=\Theta(y)\frac{by^3}{\cosh^2\left[c(\alpha y^p \ln 2)\right]} \f$
  
  class BarrierY3: public BasePotential<_Tp>
    {
    public:
      typedef BasePotential<_Tp> base_type;
      typedef typename base_type::parameter_map_type parameter_map_type;
      typedef _Tp value_type;
      typedef std::valarray<value_type> vector_type;

      BasePair<> bp;
      value_type b, c, d, alpha, y0, p, q;
      const value_type ln2=log(2); //log=natural log, i.e., ln 

    BarrierY3(void) : bp(), b(), c(), d(), alpha(), y0(), p(1), q(3)
      {
      base_type::name("barrier_y3");
      base_type::prefix(std::string());
      }

    BarrierY3(const BasePair<>& b) : bp(b), b(), c(), d(), alpha(), y0(), p(1), q(3)
      {
      base_type::name("barrier_y3");
      bp.reduce_to_smallest_symmetry();
      base_type::prefix((std::string)bp);
      }

    inline bool same_par_values(const BarrierY3 &pot) const 
      {
      return b == pot.b and c == pot.c and d == pot.d and alpha == pot.alpha and y0 == pot.y0 and p == pot.p and q == pot.q;
      }
      
    inline virtual bool get_potential_parameters(parameter_map_type& pm)
      {
      bool modified_b     = this->define_parameter(pm,"b",b);
      bool modified_c     = this->define_parameter(pm,"c",c);
      bool modified_d     = this->define_parameter(pm,"d",d);
      bool modified_alpha = this->define_parameter(pm,"alpha",alpha);
      bool modified_y0    = this->define_parameter(pm,"y0",y0);
      bool modified_p     = this->define_parameter(pm,"p",p);
      bool modified_q     = this->define_parameter(pm,"q",q);
      return modified_b || modified_c || modified_d || modified_alpha || modified_y0 || modified_p || modified_q;
      }

    inline bool parameters(const BarrierY3<value_type>& by3)
      {
      bool modified=(b !=by3.b) || (c != by3.c) || (d != by3.d) || (alpha != by3.alpha) || (y0 != by3.y0) || (p != by3.p) || (q != by3.q);
      b=by3.b; c=by3.c; d=by3.d; alpha=by3.alpha; y0=by3.y0; p=by3.p; q=by3.q;
      return modified;
      }
      
    inline virtual value_type evaluate_xy(value_type x, value_type y=value_type()) const
      {
      if (x < y0) return 0;
      value_type Theta=1;
      if (x == y0) Theta=0.5;
                                                   
      return Theta * ( b*pow(x,q)/pow(cosh(c*(alpha*pow(x,p) - d*ln2)),2) );
      }
    };

  template<class _Tp=double>
  class MorseBarrierY3: public CmbPotential<Morse<_Tp>,BarrierY3<_Tp> >
    {
    typedef  CmbPotential<Morse<_Tp>,BarrierY3<_Tp> > base_type;
    public:
      typedef _Tp value_type;
      MorseBarrierY3(void): CmbPotential<Morse<_Tp>,BarrierY3<_Tp> >() 
       {} 
      MorseBarrierY3(const BasePair<>& b): CmbPotential<Morse<_Tp>,BarrierY3<_Tp> >(b) 
       {} 
    };
    
      template<class _Tp=double>
  /// \brief A potential proportional to y^4 from peyrard09b, some authors call it a hump-potential
  ///
  /// See documentation for the equations
  class Hump: public BasePotential<_Tp>
    {
    public:
      typedef BasePotential<_Tp> base_type;
      typedef typename base_type::parameter_map_type parameter_map_type;
      typedef _Tp value_type;
      typedef std::valarray<value_type> vector_type;

      BasePair<> bp;
      value_type D, E, lambda /* -> 1/alpha */, gamma /* -> 1/beta */;
    
    private:
      value_type a, b, c, A; //determined from D, E, alpha, beta
      
    public:
 
    Hump(void) : bp(), D(), E(), lambda(), gamma(), a(), b(), c(), A()
      {
      base_type::name("hump");
      base_type::prefix(std::string());
      }

    Hump(const BasePair<>& b) : bp(b), D(), E(), lambda(), gamma(), a(), b(), c(), A()
      {
      base_type::name("hump");
      bp.reduce_to_smallest_symmetry();
      base_type::prefix((std::string)bp);
      }
      
    inline bool same_par_values(const Hump &pot) const 
      {
      return D == pot.D and E == pot.E and lambda == pot.lambda and gamma == pot.gamma;
      }
      
    inline virtual bool get_potential_parameters(parameter_map_type& pm)
      {
      bool modified_D     = this->define_parameter(pm,"D",D);
      bool modified_E     = this->define_parameter(pm,"E",E);
      bool modified_lambda = this->define_parameter(pm,"lambda",lambda);
      bool modified_gamma = this->define_parameter(pm,"gamma",gamma);
      
      calculate_factors();
      
      return modified_D || modified_E || modified_lambda || modified_gamma;
      }

    inline bool parameters(const Hump<value_type>& hmp)
      {
      bool modified=(D !=hmp.D) || (E != hmp.E) || (lambda != hmp.lambda) || (gamma != hmp.gamma);
      D=hmp.D; E=hmp.E; lambda=hmp.lambda; gamma=hmp.gamma;
      
      calculate_factors();
      
      return modified;
      }
      
    inline virtual value_type evaluate_xy(value_type x, value_type y=value_type()) const
      {
      if (x < 0.0) return  A * pow( (exp(-x/lambda) -1), 2);

      if ((x >= 0.0) and (x <= 1.0)) return a * pow(x,2) + b * pow(x,3) + c * pow(x,4);
      
      return D + E * exp(-x/gamma) * (x + gamma); //x > 1
      }
      
    private:
    //slade13phd
    inline void calculate_factors(void)
      {
      value_type alpha = 1.0/lambda, beta = 1.0/gamma;
      
      a = 6*D + 0.5*E*exp(-beta)*(pow(beta,2) + 5*beta + 12/beta + 12);
      
      b = -8*D - E*exp(-beta) * (pow(beta,2) + 4*beta + 8/beta + 8);
      
      c = 3*D + 0.5*E*exp(-beta) * (pow(beta,2) + 3*beta + 6/beta + 6);
      
      A = a / pow(alpha,2);
      
      CERR_DEBUG(DMOP_HCF) << "alpha=" << alpha << " beta=" << beta << " a=" << a << " b=" << b << " c=" << c << " A=" << A << std::endl;
      }

    };

      template<class _Tp=double>
  /// \brief A potential proportional to y^4 from peyrard09b, some authors call it a hump-potential
  ///
  /// See documentation for the equations
  class Gaussian: public BasePotential<_Tp>
    {
    public:
      typedef BasePotential<_Tp> base_type;
      typedef typename base_type::parameter_map_type parameter_map_type;
      typedef _Tp value_type;
      typedef std::valarray<value_type> vector_type;

      BasePair<> bp;
      value_type G, y0, b;
     
    Gaussian(void) : bp(), G(), y0(), b()
      {
      base_type::name("gaussian");
      base_type::prefix(std::string());
      }

    Gaussian(const BasePair<>& b) : bp(b), G(), y0(), b()
      {
      base_type::name("gaussian");
      bp.reduce_to_smallest_symmetry();
      base_type::prefix((std::string)bp);
      }
      
    inline bool same_par_values(const Gaussian &pot) const 
      {
      return G == pot.G and y0 == pot.y0 and b == pot.b;
      }
      
    inline virtual bool get_potential_parameters(parameter_map_type& pm)
      {
      bool modified_G   = this->define_parameter(pm,"G",G);
      bool modified_y0  = this->define_parameter(pm,"y0",y0);
      bool modified_b   = this->define_parameter(pm,"b",b);
      
      return modified_G || modified_y0 || modified_b;
      }

    inline bool parameters(const Gaussian<value_type>& gauss)
      {
      bool modified=(G !=gauss.G) || (y0 != gauss.y0) || (b != gauss.b);
      G=gauss.G; y0=gauss.y0; b=gauss.b;
      
      return modified;
      }
      
    inline virtual value_type evaluate_xy(value_type x, value_type y=value_type()) const
      {
      return G * exp(-pow((x-y0),2)/b);
      }
      
    };
        
  template<class _Tp=double>
  class MorseGaussian: public CmbPotential<Morse<_Tp>,Gaussian<_Tp> >
    {
    typedef  CmbPotential<Morse<_Tp>,Gaussian<_Tp> > base_type;
    public:
      typedef _Tp value_type;
      MorseGaussian(void): CmbPotential<Morse<_Tp>,Gaussian<_Tp> >() 
       {} 
      MorseGaussian(const BasePair<>& b): CmbPotential<Morse<_Tp>,Gaussian<_Tp> >(b) 
       {} 
    };
        
  template<class _Tp=double>
  /// \brief 3D Morse potential see leal20 10.1016/j.cplett.2020.137781
  class Morse3D: public BasePotential<_Tp>
    {
    public:
      typedef BasePotential<_Tp> base_type;
      typedef typename base_type::parameter_map_type parameter_map_type;
      typedef _Tp value_type;
      typedef std::valarray<value_type> vector_type;

      BasePair<> bp;
      value_type lambda; ///< \f$ \lambda_a \f$
      value_type D;      ///< \f$ D \f$, depth of the potential
      value_type R0;     ///< \f$ R_0 \f$, equilibrium distance

    Morse3D(void) : bp(), lambda(), D(), R0()
      {
      base_type::name("morse3D");
      base_type::prefix(std::string());
      }

    Morse3D(const BasePair<>& b) : bp(b), lambda(), D(), R0() 
      {
      base_type::name("morse3D");
      bp.reduce_to_smallest_symmetry();
      base_type::prefix((std::string)bp);
      }

    inline bool same_par_values(const Morse3D &pot) const 
      {
      return lambda == pot.lambda and D == pot.D and R0 == pot.R0;
      }
      
    /// \brief Retrieve the parameter from the map
    inline virtual bool get_potential_parameters(parameter_map_type& pm)
      {
      bool modified_lambda = this->define_parameter(pm,"lambda",lambda);
      bool modified_D      = this->define_parameter(pm,"D",D);
      bool modified_R0     = this->define_parameter(pm,"R0",R0);
      return modified_lambda || modified_D || modified_R0;
      }

    inline bool parameters(const Morse3D<value_type>& ms)
      {
      bool modified=(lambda != ms.lambda) || (D != ms.D) || (R0 != ms.R0);
      lambda=ms.lambda; 
      D=ms.D;
      R0=ms.R0;
      return modified;
      }

    inline virtual value_type evaluate_xy(value_type x, value_type y=value_type()) const
      {
      return D*pow(exp(-(x-R0)/lambda)-1.0,2); //leal19arxiv Eq. (6)
      }
    };
    

  template<class _Tp=double>
  class HarmonicStacking: public BasePotential<_Tp>
    {
    public:
      typedef BasePotential<_Tp> base_type;
      typedef typename base_type::parameter_map_type parameter_map_type;
      typedef _Tp value_type;
      typedef std::valarray<value_type> vector_type;

      BasePairNeighbours<> bpn;
      StrandPair<> spm, spc;
      value_type theta, k, ku, kv;

    HarmonicStacking(void) : bpn(), spm(StrandPair<>::simplify_symmetry), spc(StrandPair<>::simplify_symmetry), 
                             theta(), k(), ku(), kv()
      {
      base_type::name("harmonic");
      base_type::prefix(std::string());
      }

    HarmonicStacking(const BasePairNeighbours<>& b) : bpn(b), spm(StrandPair<>::simplify_symmetry), spc(StrandPair<>::simplify_symmetry), 
                                                      theta(), k(), ku(), kv()
      {
      base_type::name("harmonic");
      bpn.strands(spm,spc);
      bpn.reduce_to_smallest_symmetry();
      base_type::prefix((std::string)bpn);
      }

    inline bool same_par_values(const HarmonicStacking &pot) const 
      {
      return theta == pot.theta and k == pot.k and ku == pot.ku and kv == pot.kv;
      }
      
    inline virtual bool get_potential_parameters(parameter_map_type& pm)
      {
      bool found_ku=false, found_kv=false, modified_ku=false, modified_kv=false, modified_k, modified_theta;
      bool complain=false;
      //Searches first for ku and kv without complaining
      found_ku = this->find_pfx_parameter(pm,(std::string)spm,"ks",ku,modified_ku,complain);
      found_kv = this->find_pfx_parameter(pm,(std::string)spc,"ks",kv,modified_kv,complain);
      if (found_ku and found_kv)
        {
        value_type modk = ku*kv/(ku+kv);
        modified_k = modk != k;
        k = modk;
        }
      else 
        {
        modified_k = this->define_parameter(pm,"k",k);
        }
      modified_theta = this->define_parameter(pm,"theta",theta);
      return modified_k || modified_theta || modified_ku || modified_kv;
      }

    inline bool parameters(const HarmonicStacking<value_type>& hs)
      {
      bool modified=(theta != hs.theta) || (k != hs.k) || (ku != hs.ku) || (kv != hs.kv);
      theta=hs.theta; 
      k=hs.k;
      ku=hs.ku;
      kv=hs.kv;
      return modified;
      }

    inline virtual value_type evaluate_xy(value_type x, value_type y=value_type()) const
      {
      return 0.5*k*(pow(x,2)-2.0*x*y*cos(theta)+pow(y,2));
      }
    };
    
  //See leal20 10.1016/j.cplett.2020.137781
  template<class _Tp=double>
  class HarmonicStacking3DA: public BasePotential<_Tp>
    {
    public:
      typedef BasePotential<_Tp> base_type;
      typedef typename base_type::parameter_map_type parameter_map_type;
      typedef _Tp value_type;
      typedef std::valarray<value_type> vector_type;

      BasePairNeighbours<> bpn;
      value_type omega, k, J0;
      int J_order; //Order of the \Delta Z expansion Eq (10) leal19arxiv

    HarmonicStacking3DA(void) : bpn(), omega(), k(), J0(), J_order(1)
      {
      base_type::name("harmonic3DA");
      base_type::prefix(std::string());
      }

    HarmonicStacking3DA(const BasePairNeighbours<>& b) : bpn(b), omega(), k(), J0(), J_order(1)
      {
      base_type::name("harmonic3DA");
      bpn.reduce_to_smallest_symmetry();
      base_type::prefix((std::string)bpn);
      }

    inline bool same_par_values(const HarmonicStacking3DA &pot) const 
      {
      return omega == pot.omega and k == pot.k and J0 == pot.J0 and J_order == pot.J_order;
      }
      
    inline virtual bool get_potential_parameters(parameter_map_type& pm)
      {
      bool modified_k, modified_omega, modified_J0, modified_J_order;
      bool complain=false;
      modified_k       = this->define_parameter(pm,"k",k);
      modified_omega   = this->define_parameter(pm,"omega",omega);
      modified_J0      = this->define_parameter(pm,"J0",J0);
      
      value_type J_order_d=J_order;
      modified_J_order = this->define_parameter(pm,"J_order",J_order_d);
      J_order=round(J_order_d);
      std::cout << "J_order=" << J_order << std::endl;
      return modified_k || modified_omega || modified_J0 || modified_J_order;
      }

    inline bool parameters(const HarmonicStacking3DA<value_type>& hs)
      {
      bool modified=(omega != hs.omega) || (k != hs.k) || (J0 != hs.J0) || (J_order != hs.J_order);
      omega=hs.omega; 
      k=hs.k;
      J0=hs.J0;
      J_order=hs.J_order;
      return modified;
      }

    inline virtual value_type evaluate_xy(value_type x, value_type y=value_type()) const
      {  
      value_type res=value_type();
      switch (J_order)
        {
        case 1: res  =   -k/(8  *pow(J0,2)) * pow( ( pow(x-y,2) + pow(omega,2)*x*y ), 2); 
                break;

        case 2: res  =   -k/(8  *pow(J0,2)) * pow( ( pow(x-y,2) + pow(omega,2)*x*y ), 2)
                         +k/(16 *pow(J0,4)) * pow( ( pow(x-y,2) + pow(omega,2)*x*y ), 3)
                       -5*k/(128*pow(J0,6)) * pow( ( pow(x-y,2) + pow(omega,2)*x*y ), 4); //leal19arxivsupp Eq(S4)
                break;
                
        
        default: 
          std::cerr << "Error: Delta z expansion of harmonic3DA.J_order=" << J_order << " not implemented" << std::endl;
          exit(1);
        }
        //  exp(-Beta*evaluate_xy(x[i],y[j])); Already has the minus sign, we return sign-inverted
      return -res;  
      }
    };    

  template<class _Tp=double>
  class AnharmonicStacking: public BasePotential<_Tp>
    {
    public:
      typedef BasePotential<_Tp> base_type;
      typedef typename base_type::parameter_map_type parameter_map_type;
      typedef _Tp value_type;
      typedef std::valarray<value_type> vector_type;

      BasePairNeighbours<> bpn;
      value_type rho, alpha;
      HarmonicStacking<value_type> harmonic;

    AnharmonicStacking(void) : bpn(), rho(), alpha(), harmonic()
      {
      base_type::name("anharmonic");
      base_type::prefix(std::string());
      }

    AnharmonicStacking(const BasePairNeighbours<>& b) : bpn(b), rho(), alpha(), harmonic(b)
      {
      base_type::name("anharmonic");
      bpn.reduce_to_smallest_symmetry();
      base_type::prefix((std::string)bpn);
      }

    inline bool same_par_values(const AnharmonicStacking &pot) const 
      {
      return rho == pot.rho and alpha == pot.alpha and harmonic.same_par_values(pot.harmonic);
      }
      
    inline virtual bool get_potential_parameters(parameter_map_type& pm)
      {
      bool modified_rho      = this->define_parameter(pm,"rho",rho);
      bool modified_alpha    = this->define_parameter(pm,"alpha",alpha);
      bool modified_harmonic = harmonic.get_potential_parameters(pm);
      base_type::ParMap.insert(harmonic.ParMap.begin(),harmonic.ParMap.end());
      return modified_rho || modified_alpha || modified_harmonic;
      }

    inline bool parameters(const AnharmonicStacking<value_type>& ast)
      {
      bool modified_harmonic = harmonic.parameters(ast.harmonic);
      bool modified = (rho != ast.rho) || (alpha != ast.alpha);
      rho=ast.rho; alpha=ast.alpha;
      return modified || modified_harmonic;
      }

    inline virtual value_type evaluate_xy(value_type x, value_type y=value_type()) const
      {return (1.0+rho*exp(-alpha*(x+y)))*harmonic.evaluate_xy(x,y);}
    };

  template<class _Tp=double>
  class ExactStacking: public BasePotential<_Tp>
    {
    public:
      typedef BasePotential<_Tp> base_type;
      typedef typename base_type::parameter_map_type parameter_map_type;
      typedef _Tp value_type;
      typedef std::valarray<value_type> vector_type;

      BasePairNeighbours<> bpn;
      value_type h;
      HarmonicStacking<value_type> harmonic; // Here we also use the parameters from Morse

    ExactStacking(void) : bpn(), h(), harmonic()
      {
      base_type::name("exactstack");
      base_type::prefix(std::string());
      }

    ExactStacking(const BasePairNeighbours<>& b) : bpn(b), h(), harmonic(b)
      {
      base_type::name("exactstack");
      bpn.reduce_to_smallest_symmetry();
      base_type::prefix((std::string)bpn);
      }

    inline bool same_par_values(const ExactStacking &pot) const 
      {
      return h == pot.h and harmonic.same_par_values(pot.harmonic);
      }
      
    inline virtual bool get_potential_parameters(parameter_map_type& pm)
      {
      bool modified_h        = this->define_parameter(pm,"h",h);
      bool modified_harmonic = harmonic.get_potential_parameters(pm);// Here we also use the parameters from Morse
      base_type::ParMap.insert(harmonic.ParMap.begin(),harmonic.ParMap.end());
      return modified_h || modified_harmonic;
      }

    inline bool parameters(const ExactStacking<value_type>& est)
      {
      bool modified_harmonic = harmonic.parameters(est.harmonic);
      bool modified =(h != est.h);
      h=est.h;
      return modified || modified_harmonic;
      }

    inline virtual value_type evaluate_xy(value_type x, value_type y=value_type()) const
      {return harmonic.k * pow( sqrt(pow(h,2) + 0.5*(pow(x,2)-2.0*x*y*cos(harmonic.theta)+pow(y,2))) -h ,2);} // This is where the equation is calculated, x stands for y_n and y stands for y_{n-1}
    };

  template<class _Tp=double>
  class FiniteEnthalpyStacking: public BasePotential<_Tp>
    {
    public:
      typedef BasePotential<_Tp> base_type;
      typedef typename base_type::parameter_map_type parameter_map_type;
      typedef _Tp value_type;
      typedef std::valarray<value_type> vector_type;

      BasePairNeighbours<> bpn;
      value_type DeltaH, b, kb, C;

    FiniteEnthalpyStacking(void) : bpn(), DeltaH(), b(), kb()
      {
      base_type::name("finite_enthalpy");
      base_type::prefix(std::string());
      }

    FiniteEnthalpyStacking(const BasePairNeighbours<>& bp) : bpn(bp), DeltaH(), b(), kb()
      {
      base_type::name("finite_enthalpy");
      bpn.reduce_to_smallest_symmetry();
      base_type::prefix((std::string)bpn);
      }

    inline bool same_par_values(const FiniteEnthalpyStacking &pot) const 
      {//DeltaH, b, kb, C
      return DeltaH == pot.DeltaH and b == pot.b and kb == pot.kb and C == pot.C;
      }
      
    inline virtual bool get_potential_parameters(parameter_map_type& pm)
      {
      bool modified_DeltaH = this->define_parameter(pm,"DeltaH",DeltaH);
      bool modified_b      = this->define_parameter(pm,"b",b);
      bool modified_kb     = this->define_parameter(pm,"kb",kb);
      bool modified_C      = this->define_parameter(pm,"C",C); //see joyeux09 Eq. 2.1, originally C=2.0
      return modified_DeltaH || modified_b || modified_kb || modified_C;
      }

    inline bool parameters(const FiniteEnthalpyStacking<value_type>& ast)
      {
      bool modified;
      modified = (DeltaH != ast.DeltaH) || (b != ast.b) || (kb != ast.kb) || (C != ast.C);
      DeltaH=ast.DeltaH; b=ast.b; kb=ast.kb; C=ast.C;
      return modified;
      }

    inline virtual value_type evaluate_xy(value_type x, value_type y=value_type()) const
      {
      value_type xy2=pow(x-y,2);
      return DeltaH/C*(1.0-exp(-b*xy2))+kb*xy2;
      }
    };

  template<class _Tp=double>
  class Yukawa: public BasePotential<_Tp>
    {
    public:
      typedef BasePotential<_Tp> base_type;
      typedef typename base_type::parameter_map_type parameter_map_type;
      typedef _Tp value_type;
      typedef std::valarray<value_type> vector_type;
      value_type k, lambda;

    Yukawa(std::string p=std::string()) : k(), lambda()
      {
      base_type::name("yukawa");
      base_type::prefix(p);
      }

    inline bool same_par_values(const Yukawa &pot) const 
      {//k, lambda
      return  k == pot.k and lambda == pot.lambda;
      }

    inline virtual bool get_potential_parameters(parameter_map_type& pm)
      {
      bool modified_k      = this->define_parameter(pm,"k",k);
      bool modified_lambda = this->define_parameter(pm,"lambda",lambda);
      return modified_k || modified_lambda;
      }

    inline bool parameters(const Yukawa<value_type>& hs)
      {
      bool modified=(lambda != hs.lambda) || (k != hs.k);
      lambda=hs.lambda; 
      k=hs.k;
      return modified;
      }

    inline virtual value_type evaluate_xy(value_type x, value_type y=value_type()) const
      {
      return k*exp(-fabs(x)/lambda)/fabs(x);
      }
    };
  
  };

#endif
