// $Id: Legendre.h 1247 2021-12-13 18:08:41Z ge $
/// \file Legendre.h
/// \brief contains the Legendre function class
///
/// $Revision: 1247 $
/// \author Gerald Weber <gweberbh@gmail.com>
#ifndef GBC_LEGENDRE_H
#define GBC_LEGENDRE_H "$Id: Legendre.h 1247 2021-12-13 18:08:41Z ge $"
#include <limits>
#include <fstream>
#include <gsl/gsl_sf_legendre.h>
#include <cmath>
#include "MathAux.h"
#include <boost/filesystem.hpp>
#include "ErrorCodes.h"

namespace gbc
  {
  template<class _VectorTp  =std::valarray<double> >
  
  /// \brief Class which calculates Gauss-Legendre integration weights
  ///
  /// \attention This class needs the GNU Scientific Library (GSL) 
  class Legendre
    {
    public:
    typedef typename _VectorTp::value_type value_type;
    typedef _VectorTp vector_type;
    typedef std::pair<value_type,value_type> bracket_pair; ///< Type of values for which a sign change was detected
    typedef std::vector<bracket_pair> bracket_vector;
    typedef int order_type;

    std::string Base_directory; ///< Base directory where calculated roots and weights are stored.
    std::string Base_name;      ///< Base name of the saved calculations.
    std::string Base_ext;       ///< Extension of the saved calculations.
    bool Load_calculations;     ///< If true, load previous calculations if they exist in base_directory.
    bool Load_Done;             ///< If true, loading previous calculations was OK.
    bool Save_calculations;     ///< If true, save calculations in base_directory.
    bool Roots_calculated;      ///< Signals if the calculation was already performed for given parameters.
    bool Debug;                 ///< If true print our Debug information

    order_type Order;           ///< Order of the Legendre polynomial.
    value_type Root_limit;      ///< Limit of the root find calculation.

    long int steps_used;        ///< Steps effectively used for finding all roods.
    bracket_vector bracket;     ///< Holds all detected sign changes.
    vector_type Roots;          ///< Calculated or loaded roots.
    vector_type Weights;        ///< Calculated or loaded integration weigths.
    value_type Scale_weights;   ///< Number by which the weights need to be scaled

    Legendre(void)
      :Base_directory(std::string()), Base_name(std::string("gauss-legendre-")), Base_ext(".dat"),
       Load_calculations(false), Load_Done(false), Save_calculations(false), Roots_calculated(false), 
       Debug(false), Order(0), Scale_weights(1.0) {};

    Legendre(const std::string bd,
             const std::string bn=std::string("gauss-legendre-"),
             const std::string be=std::string(".dat"),
             const bool load_calc=true,
             const bool save_calc=true)
      :Base_directory(bd), Base_name(bn), Base_ext(be), 
       Load_calculations(load_calc), Load_Done(false), Save_calculations(save_calc), Roots_calculated(false), 
       Debug(false), Order(0), Scale_weights(1.0) {};

    /// \brief Evaluates the Legendre Polynomial of $l$-th order at x
    inline value_type evaluate(const order_type l, ///< order of the polynomial
                               const value_type x) ///< function argument
      {return gsl_sf_legendre_Pl(l,x) ;}

    /// \brief Evaluates the Legendre Polynomial of $l$-th order at positions in vector $x$
    inline vector_type evaluate(const order_type l,   ///< order of the polynomial
                                const vector_type &x) ///< function argument
      {
      long int i,m=x.size();
      vector_type result(m);
      for (i=0; i < m; i++) {result[i]=gsl_sf_legendre_Pl(l,x[i]);}
      return result;
      }
  
    /// \brief Finds the roots in the interval $(-1,1)$.
    inline void find_roots(const int& order,          ///< order of the polynomial 
                           const value_type& limit)   ///< root find limit
      {
      unsigned long int i,j,start_steps=10,steps=order*order/4;
      size_t expected_roots=order/2;
      bracket.reserve(order/2+1);
      value_type max=1;
      while ((bracket.size() < expected_roots) && (steps < std::numeric_limits<unsigned long int>::max()))
        {
        bracket.clear();
	steps*=2;
	vector_type linspace=gbc::linear_space<vector_type>(0.0,max,steps);
        vector_type legendre=evaluate(order,linspace);
        for (i=0; i < (steps -1); ++i)
          {
	  if (legendre[i]*legendre[i+1] < 0) 
            {bracket.push_back(bracket_pair(linspace[i],linspace[i+1]));}
	  }
 	}
      steps_used=steps;

      int nroot=0;
      vector_type found_roots(bracket.size());
      if (2*(order/2)!=order) 
        {
        found_roots.resize(bracket.size()+1);
        found_roots[nroot++]=0.0;
        }
      for(i=0; i < bracket.size(); ++i)
        {
	value_type zero=std::numeric_limits<value_type>::max();
	steps=start_steps;
        while ((zero > limit) && (steps < std::numeric_limits<unsigned long int>::max()))
	  {
	  vector_type linspace=gbc::linear_space<vector_type>(bracket[i].first,bracket[i].second,steps);
	  vector_type legendre=evaluate(order,linspace);
          for (j=0; j < steps; j++)
            {
	    if (legendre[j]*legendre[j+1] < 0) 
	      {bracket[i].first=linspace[j]; bracket[i].second=linspace[j+1]; break;}
	    }
	  zero=fabs(evaluate(order,(bracket[i].first+bracket[i].second)/2.0));
	  }
	 found_roots[nroot++]=(bracket[i].first+bracket[i].second)/2.0;
        }

      unsigned long int sz=found_roots.size();
      int offset=(order%2); 
      //std::cout << order << " " << offset << " " << sz << std::endl;
      Roots.resize(order);
      for (i=0; i < sz; ++i)
        {Roots[i]=-found_roots[sz-i-1];}
      for (i=sz; i < 2*sz-offset; ++i)
        {Roots[i]=found_roots[i-sz+offset];}
      Roots_calculated=true;
      }
      
    inline std::string filename(void)
      {
      std::stringstream fname;
      if (!Base_directory.empty()) 
        {
	boost::filesystem::create_directory(Base_directory);
        fname << Base_directory << "/";
	}
      fname << Base_name << Order << Base_ext;
      return fname.str();
      }

    inline void load_calculations(void)
      {
      Load_Done=false;
      if (boost::filesystem::exists(filename())) 
        {
	CERR_DDEBUG(DLEG_LOADF) << "Loading [" << filename().c_str() << "]" << std::endl;

        std::ifstream sl;
        sl.open(filename().c_str());
        if (!sl.good()) 
           {
           return;
           }
        unsigned int count=0; 
        Roots.resize(Order); Weights.resize(Order);
        while (!sl.eof())
          {
          typename vector_type::value_type r,w; 
          sl >> r >> w;
          if (!sl.eof()) {Roots[count]=r; Weights[count]=w; count++;}
          }
        sl.close();
        if ((Roots.size() == count) && (Weights.size() == count)) Load_Done=true;
        }
      else
        {
	CERR_DDEBUG(DLEG_FNF) << "File [" << filename().c_str() << "] not found" << std::endl;
	}
      }

    inline void save_calculations(void)
      {
      std::ofstream sl;
      sl.open(filename().c_str()); 
      CERR_DDEBUG(DLEG_SAVFI) << "Saving [" << filename().c_str() << "]" << std::endl;
      int count=0, sz=Roots.size();
      sl.precision(20);
      for (count=0; count < sz; count++)
        {
        sl << std::fixed << Roots[count] << " " << Weights[count] << std::endl;
        }
      sl.close();
      }

    /// \brief All roots in the interval $(-1,1)$.
    ///
    /// If roots are already calculated restore them or load them.
    /// If a new calculation is needed, save the results in a file.
    inline vector_type roots(const int order, const value_type limit)
      {
      if (Order!=order)
        {
        Load_Done=Roots_calculated=false; Order=order;
        } 
      if (!Load_Done && Load_calculations)
        {load_calculations();}
      
      Root_limit=limit;
      
      if (!Load_Done && !Roots_calculated)
        {
        find_roots(order,limit);
        integration_weights(order,limit);
        if (Save_calculations) save_calculations();
        }
      
      return Roots;
      }

    inline vector_type weights(void)
      {
      vector_type w=Scale_weights * Weights;
      return w;
      }

    /// \brief calculates the integration weigths 
    inline vector_type integration_weights(const int order,        ///< order of the polynomial 
                                           const value_type limit) ///< root find limit
      {
      roots(order,limit);
      vector_type pl=evaluate(order+1,Roots);
      long int i,sz=Roots.size();
      Weights.resize(sz);
      for (i=0; i < sz; i++)
        {
        Weights[i]=2.0*(1.0-std::pow(Roots[i],2))/((order+1)*(order+1)*std::pow(pl[i],2));
        }
      return Weights;
      }

    inline void adjust_space(vector_type& x)
      {
      size_t N=x.size();
      typename vector_type::value_type xm,xl;
      xm=(x[N-1]+x[0])/2.0;
      xl=(x[N-1]-x[0])/2.0;
      x=xm+xl*Roots;
      Scale_weights= xl;
      }
  };
}
#endif
