// $Id: Section.h 628 2011-08-31 15:42:16Z ge $
/// \file Section.h
/// \brief Section template class
///
/// $Revision: 628 $
/// \author Gerald Weber

#ifndef SECTION_H
#define SECTION_H "$Id: Section.h 628 2011-08-31 15:42:16Z ge $"
#include <iostream>


namespace gbc
 {
  template<class _Tp>
  /// \brief Section template class.
  ///
  /// This class inherits pair, therefore the begining can be accessed with Section->first
  /// and the end with Section->second.
  class Section: public std::pair<_Tp,_Tp>
    {
    public:
      Section(void){}

      /// \brief Sets the begining of a Section.
      inline void begin(const _Tp &b) ///< the section start
        {this->first=b;}

      /// \brief Gets the begining of a Section.
      inline _Tp  begin(void) {return this->first;}

      /// \brief Sets the end of a Section.
      inline void end(const _Tp &e)  ///< the section end
        {this->second=e;}

      /// \brief Gets the end of a Section.
      inline _Tp end(void) {return this->second;}

      /// \brief The size of a Section.
      ///
      /// This will be end()-begin()+1
      inline _Tp size(void) {return (end()-begin()+static_cast<_Tp>(1));}

      /// \brief Verifies if the value lies within or at the section boundary.
      ///
      /// \attention This methods returns true including at exactly the beginning/end.
      inline bool is_inside(const _Tp &in) ///< the position to be checked
        {
	return ((in >= this->first) && (in <= this->second));
	}

      /// \brief Sets the begining and the end of a Section.
      inline void section(const _Tp &b, ///< the begining
                          const _Tp &e) ///< the end
        {
	begin(b);
	if (b > e ) end(b); else end(e);
	}

      /// \brief Sets the begining and the end with the same value.
      ///
      /// This methods defines a Section of size 1.
      inline void section(const _Tp &b) ///< sets the begining/end
         {section(b,b);}

     /// \brief Extractor which prints the section values.
     ///
     /// \attention Note that extractor << needs to be defined for _Tp.
     friend std::ostream& operator<<(std::ostream &out, ///< the stream to write to
                                     const Section &sec)///< the Section to display
       {
       out << sec.first << " " << sec.second;
       return out;
       }

     /// \brief Inserter which gets the section values.
     ///
     /// \attention Note that inserter >> needs to be defined for _Tp.
     friend std::istream& operator>>(std::istream &in, ///< the stream from which to get the data
                                     Section &sec)     ///< the Section which gets the data
       {
       in >> sec.first >> sec.second;
       return in;
       }
   };
  };

#endif


