// $Id: EigenSystem.h 1247 2021-12-13 18:08:41Z ge $
/// \file EigenSystem.h
/// \brief contains the EigenSystem function class
///
/// $Revision: 1247 $
/// \author Gerald Weber <gweberbh@gmail.com>
#ifndef GBC_EIGENSYSTEM_H
#define GBC_EIGENSYSTEM_H "$Id: EigenSystem.h 1247 2021-12-13 18:08:41Z ge $"
#include "valmatrix.h"
#include <complex>
#include <cmath>
#include "ErrorCodes.h"
//#include <f2c.h>
//SUBROUTINE DSYEV(JOBZ, UPLO, N, A, LDA, W, WORK, LWORK, INFO ) 
//CHARACTER JOBZ, UPLO
//INTEGER INFO, LDA, LWORK, N
//DOUBLE PRECISION A( LDA, * ), W( * ), WORK( * )
extern "C" void dsyev_(char*,char*,int*,double*,int*,double*,double*,int*,int*);
extern "C" void ssyev_(char*,char*,int*,float*,int*,float*,float*,int*,int*);

//SUBROUTINE DGEEV(JOBVL, JOBVR, N, A, LDA, WR, WI, VL, LDVL, VR, LDVR, WORK, LWORK, INFO ) 
//CHARACTER JOBVL, JOBVR 
//INTEGER INFO, LDA, LDVL, LDVR, LWORK, N 
//DOUBLE PRECISION A( LDA, * ), VL( LDVL, * ), VR( LDVR, * ), WI( * ), WORK( * ), WR( * ) 
extern "C" void dgeev_(char* JOBVL,char* JOBVR,int* N,double* A,int* LDA,double* WR,double* WI,double* VL,int* LDVL,double* VR,int* LDVR,double* WORK,int* LWORK,int* INFO);

using namespace gbc;

namespace gbc
  {
  template<class _InputTp=double,class _OutputTp=_InputTp, 
           class _InputVectorTp=valmatrix<_InputTp>, class _OutputVectorTp=valmatrix<_OutputTp> >
  /// \brief Wrapper for Eigensystem calculation
  ///
  /// \attention This class needs the GNU Scientific Library (GSL) 
  /// file:///usr/share/doc/packages/gsl/gsl-ref_14.html#SEC234
  class EigenSystem
    {
    public:
      typedef _InputTp         input_type;
      typedef _OutputTp        output_type;
      typedef _InputVectorTp   input_vector_type;
      typedef _OutputVectorTp  output_vector_type;

      output_vector_type Eigenvector, Eigenvalue;
      bool Decrease_sort_eigenvalues;
      bool Calculate_eigenvalue_only;
      int Dimension;
      int Workspace_size;
      int Evalues_size;
      input_type* Temporary;
      input_type* Workspace;
      input_type* Evalues;
   
      EigenSystem(void)
        : Eigenvector(), Eigenvalue(),
          Decrease_sort_eigenvalues(true), Calculate_eigenvalue_only(true),
          Dimension(1), Workspace_size(1), Evalues_size(1),
          Temporary(new input_type[Dimension]), Workspace(new input_type[Workspace_size]),
          Evalues(new input_type[Evalues_size])
      {
      }
/*
   ~EigenSystem(void)
     {
     delete[] Temporary;
     delete[] Workspace;
     delete[] Evalues;
     }
*/
   void resize_temporary(const size_t& ndim)
     {
      if (static_cast<size_t>(Dimension) != ndim)
        {
        Dimension=ndim;
        delete[] Temporary;
        Temporary=new input_type[Dimension*Dimension];
        }
     }

   void resize_workspace(const size_t& ndim)
     {
      if (static_cast<size_t>(Workspace_size) != ndim)
        {
        Workspace_size=ndim;
        delete[] Workspace;
        Workspace=new input_type[Workspace_size];
        }
     }

   void resize_evalues(const size_t& ndim)
     {
      if (static_cast<size_t>(Evalues_size) != ndim)
        {
        Evalues_size=ndim;
        delete[] Evalues;
        Evalues=new input_type[Evalues_size];
        }
     }

   void diagonalize(const input_vector_type& vec) {};
   
   };


  template<>
  inline void EigenSystem<double,double>::diagonalize(const EigenSystem<double,double>::input_vector_type& vec)
    {
    resize_temporary(vec.Rows);
    resize_evalues(vec.Rows);

    for (int i=0; i < Dimension*Dimension; i++) 
      {
      if (std::isinf(vec[i]) or std::isnan(vec[i]))
        {
        CERR_ERROR(ERRCDINAN) << " Error: cannot diagonalize infinite or NaN vec[" << i << "]=" << vec[i] << std::endl
                              << "you may have to check your integration limits (option -int)" << std::endl;
        CERR_TERM
        }
      else Temporary[i]=vec[i];
      }

    char jobvl='N',uplo='U';
    if (!Calculate_eigenvalue_only) jobvl='V';

    int lwork=-1, retval;
    double work;

    // This call determines the optimal working space
    dsyev_(&jobvl,&uplo,&Dimension,Temporary,&Dimension,Evalues,&work,&lwork,&retval);
    lwork=static_cast<int>(work);
    resize_workspace(lwork);

    // Now do the eigenvalue calculation
    dsyev_(&jobvl,&uplo,&Dimension,Temporary,&Dimension,Evalues,Workspace,&lwork,&retval);
    if (retval)
      {
      CERR_ERROR(ERRPRWDIA) << "problems with diagonalization" << std::endl
                            << "retval=" << retval << std::endl
                            << "Dimension=" << Dimension << std::endl
                            << "lwork=" << lwork << std::endl
                            << vec << std::endl;
      CERR_TERM
      }

    Eigenvalue.resize(1,Dimension);
    Eigenvalue=Evalues;
    if (Decrease_sort_eigenvalues) Eigenvalue.reverse();

    Eigenvector.resize(Dimension,Dimension);
    Eigenvector=Temporary;

    Eigenvector=transpose(Eigenvector);
    if (Decrease_sort_eigenvalues) Eigenvector.reverse_columns();
    }

  template<>
  inline void EigenSystem<double,std::complex<double> >::diagonalize(const EigenSystem<double,std::complex<double> >::input_vector_type& vec)
    {
    int dimension=vec.Rows;
    double* tmp=new double[dimension*dimension];
    for (int i=0; i < dimension*dimension; i++) tmp[i]=vec[i];

    char left_eigen='N', right_eigen='N';
    if (!Calculate_eigenvalue_only) right_eigen='V';

    int lwork, retval;
    double* work = new double[1];
    double* real_evalues  = new double[dimension];
    double* imag_evalues  = new double[dimension];
    double* left_vectors  = new double[dimension*dimension];
    double* right_vectors = new double[dimension*dimension];

    // This call determines the optimal working space
    lwork=-1;
    // dgeev_(char* JOBVL,char* JOBVR,int* N,double* A,int* LDA,double* WR,double* WI,double* VL,int* LDVL,double* VR,int* LDVR,double* WORK,int* LWORK,int* INFO);
    dgeev_(&left_eigen,        //char* JOBVL
           &right_eigen,       //char* JOBVR
           &dimension,         //int* N
           tmp,                //double* A
           &dimension,         //int* LDA
           real_evalues,       //double* WR
           imag_evalues,       //double* WI
           left_vectors,       //double* VL
           &dimension,         //int* LDVL
           right_vectors,      //double* VR
           &dimension,         //int* LDVR
           work,               //double* WORK
           &lwork,             //int* LWORK
           &retval);           //int* INFO
 
    lwork=static_cast<int>(work[0]);
    delete[] work;
    work = new double[lwork];

    // Now do the eigenvalue calculation
    dgeev_(&left_eigen,        //char* JOBVL
           &right_eigen,       //char* JOBVR
           &dimension,         //int* N
           tmp,                //double* A
           &dimension,         //int* LDA
           real_evalues,       //double* WR
           imag_evalues,       //double* WI
           left_vectors,       //double* VL
           &dimension,         //int* LDVL
           right_vectors,      //double* VR
           &dimension,         //int* LDVR
           work,               //double* WORK
           &lwork,             //int* LWORK
           &retval);           //int* INFO

    if (retval < 0)
      {CERR_ERROR(ERRTHARGI) << " the" <<  retval << "-th argument had an illegal value " << std::endl; CERR_TERM;}
    if (retval > 0)
      {
      CERR_ERROR(ERRQRALFA) << "the QR algorithm failed to compute all the eigenvalues, and no eigenvectors have been computed; elements"
      << retval+1 << ":" << dimension <<  "of WR and WI contain eigenvalues which have converged." << std::endl; 
      CERR_TERM;
      }

    input_vector_type Real_eigenvalues(1,dimension);
    Real_eigenvalues=real_evalues;
    input_vector_type Imag_eigenvalues(1,dimension);
    Imag_eigenvalues=imag_evalues;

    Eigenvalue.resize(1,dimension);
    Eigenvalue.real(Real_eigenvalues);
    Eigenvalue.imag(Imag_eigenvalues);
    Eigenvector.resize(dimension,dimension);

    input_vector_type eigenmatrix(dimension,dimension);
    eigenmatrix=right_vectors;

    // C Matrix=transpose(Fortran Matrix)
    eigenmatrix=transpose(eigenmatrix);

    output_vector_type rc(dimension,1);
    for (int j=0; j < dimension-1; ++j) 
      {
      if (Eigenvalue(0,j) != conj(Eigenvalue(0,j+1)))
        {
        //If the j-th eigenvalue is real, then v(j) = VR(:,j), the j-th column of VR. 
        rc.real(eigenmatrix.column(j));
        Eigenvector.column(j,rc);
        }
      else
        {
        //If the j-th and (j+1)-st eigenvalues form a complex conjugate pair, 
        //then v(j) = VR(:,j) + i*VR(:,j+1) and 
        //v(j+1) = VR(:,j) - i*VR(:,j+1).
        rc.real(eigenmatrix.column(j));
        rc.imag(eigenmatrix.column(j+1));
        Eigenvector.column(j+1,rc);
        rc.conj();
        Eigenvector.column(j,rc);
        ++j;
        }
      }

    delete[] tmp; delete[] work; delete[] real_evalues; delete[] imag_evalues;
    delete[] left_vectors; delete[] right_vectors; 
    }

}
#endif
