// $Id: HeterogenousTM.h 1369 2024-11-29 14:20:00Z ge $
/// \file HeterogenousTM.h
/// \brief contains the class for Heterogenous Transfer Matrix calculations
///
/// $Revision: 1369 $
/// \author Gerald Weber <gweberbh@gmail.com>
#ifndef GBC_EXP_HETEROGENOUSTM_H
#define GBC_EXP_HETEROGENOUSTM_H "$Id: HeterogenousTM.h 1369 2024-11-29 14:20:00Z ge $"
#include "TransferMatrix.h"
#include "Duplex.h"
#include "SequenceInfo.h"
#include "NeighbourSequence.h"
#include <gsl/gsl_math.h>
#include <boost/filesystem.hpp>
#include "Minimization.h"
#include "ErrorCodes.h"
#undef max
#undef min

namespace gbc
  {
  template<class _Tp=double>
  class HeterogenousTM
    {
    public:
      typedef _Tp                                        value_type;
      typedef TransferMatrix<value_type>                 transfer_matrix_type;
      typedef ParameterMap<value_type>                   parameter_map_type;
      typedef std::string                                bp_id_type;
      typedef valmatrix<value_type>                      matrix_type;
      typedef std::vector<matrix_type>                   vector_matrix_type;
      enum tm_matrix_flags {A_and_C,A_only,C_only};  ///< The short identification for Hamiltonians. *** Add new identification here, 

      class GenericTransferMatrix 
        {
        public:
        transfer_matrix_type *TM;
        bool Use_transpose_matrix, Lock_TM_pointer, A_matrix_calculated, C_matrix_calculated, ADelta_matrix_calculated, Delta_matrix_calculated;
        tm_matrix_flags Matrix_calculation;
        BasePairNeighbours<> Bpn, Bpn_from;
      
        GenericTransferMatrix(void): TM(nullptr), Use_transpose_matrix(false), Lock_TM_pointer(false), 
                                     A_matrix_calculated(false), C_matrix_calculated(false), 
                                     ADelta_matrix_calculated(false), Delta_matrix_calculated(false),
                                     Matrix_calculation(C_only), Bpn(), Bpn_from()  
          {
          TM= new transfer_matrix_type();
          //std::cerr << "Creating empty matrix" << std::endl;
          };
        
        GenericTransferMatrix(const BasePairNeighbours<> &bpn): 
                                     TM(nullptr), Use_transpose_matrix(false), Lock_TM_pointer(true),
                                     A_matrix_calculated(false), C_matrix_calculated(false), 
                                     ADelta_matrix_calculated(false), Delta_matrix_calculated(false),
                                     Matrix_calculation(C_only), Bpn(bpn), Bpn_from()  
          {
          TM= new transfer_matrix_type(bpn);
          //std::cerr << "Creating matrix [" << bpn << "]" << std::endl;
          };
          
        void reset_matrix_calculation(void) 
          {
          A_matrix_calculated=C_matrix_calculated=ADelta_matrix_calculated=Delta_matrix_calculated=false;
          }
          
        void use_transpose_matrix_from(const BasePairNeighbours<> &bpn, const GenericTransferMatrix& tm)
          {
          Bpn=bpn;
          Bpn_from=tm.Bpn;
          if ( (TM != nullptr) and (not Lock_TM_pointer) ) delete TM;
          TM = tm.TM;
          Lock_TM_pointer=true;
          Matrix_calculation = tm.Matrix_calculation;
          Use_transpose_matrix=true;
          }
          
        bool get_tm_parameters(ParameterMap<value_type>& pm)
          {
          //std::cerr << *this << std::endl;
          bool modified=false;
          if (not Use_transpose_matrix) modified = TM->get_hamiltonian_parameters(pm);
          return modified;
          }
          
        friend std::ostream& operator<<(std::ostream &out, const GenericTransferMatrix& tm)
          {
          out << "Bpn=" << tm.Bpn 
              << " Matrix_calculation=" << tm.Matrix_calculation
              << " A_matrix_calculated=" << tm.A_matrix_calculated
              << " C_matrix_calculated=" << tm.C_matrix_calculated
              << " ADelta_matrix_calculated=" << tm.ADelta_matrix_calculated
              << " Delta_matrix_calculated=" << tm.Delta_matrix_calculated;
          out << " Use_transpose_matrix=" << tm.Use_transpose_matrix
              << " from=" << tm.Bpn_from; 
          return out;
          }

      
        };
          
      typedef std::map<bp_id_type,GenericTransferMatrix> tm_map_type;
      typedef std::valarray<value_type>                  vector_type;
      typedef std::map<bp_id_type,matrix_type>           matrix_map_type;
      typedef std::map<bp_id_type,tm_matrix_flags>       matrix_calculation_map_type;
      typedef std::map<bp_id_type,Range<double> >        range_map_type;
      typedef SequenceInfo<true,value_type>               sequence_info_type; //true refers to periodicity
      typedef typename sequence_info_type::duplex_ref_type duplex_ref_type;
      typedef typename sequence_info_type::neighbours_ref_type neighbours_ref_type;
      typedef std::deque<std::string>                    string_deque_type;
      typedef typename transfer_matrix_type::quadrature_type      quadrature_type;
      
    protected:
      value_type Temperature;               ///< The temperature to be used for the hamiltonian

    public:
    bool             Debug;
    bool             Periodic_boundary_condition; ///< Controls the boundary condition
    bool             Verbose;
    
    size_t           Largest_N_in_dataset;        ///< Holds the largest number of nearest-neighbours in the dataset
 
    tm_map_type      TM_map;                      ///< Map of TransferMatrices
    range_map_type   Range_map;

    std::string      ParameterIdName;
    std::string      MatrixDir;

    matrix_map_type  C_matrix_map;                ///< Map of \f$ C \f$ matrices
    matrix_map_type  A_matrix_map;                ///< Map of \f$ A \f$ matrices
    matrix_map_type  Delta_matrix_map;            ///< Map of \f$ \Delta \f$ matrices
    matrix_map_type  ADelta_matrix_map;           ///< Map of \f$ \Delta \f$ matrices
    matrix_map_type* Loop_matrix_map;
    matrix_type      Y;                           ///< The \f$ Y \f$ matrix
    matrix_type      Eigenvector;
    matrix_type      Eigenvalue;
    matrix_type      Lambda_matrix;               ///< The \f$ \Lambda \f$ matrix
    vector_matrix_type Lambda_matrix_pow;         ///< Holds Lambda_matrix^n
    vector_matrix_type Gamma;                     ///< The Gamma matrix used in the partition function expansion
    value_type       Zy;                          ///< The Zy part of the partition function
    value_type       Fy;                          ///< Helmoltz free energy of Zy
    value_type       Mu_step;                     ///< Step of the gaussian fitting
    
    bool             Y_recalculate;
    bool             Always_recalculate_matrices; 
    
    typename tm_map_type::iterator Base;          ///< Iterator which points to the expansion base
    bp_id_type                     BaseName;      ///< Name of the base pair neighbours on which the expansion take place

    parameter_map_type parameter_map;


    sequence_info_type*  pSI;
    bool                 SI_local;                ///< true if pSI was locally defined with new operator
    quadrature_type*     Quadrature;              ///< The quadrature derived from TransferMatrix

    HeterogenousTM(void): Temperature(), Periodic_boundary_condition(false), Debug(false),
                          Largest_N_in_dataset(),
                          Zy(), Fy(), Mu_step(0.001),
                          Y_recalculate(true), Always_recalculate_matrices(false),
                          pSI(), SI_local(false)
      {
      Quadrature = new quadrature_type();
      Quadrature->Rule.Save_calculations=true;
      Quadrature->Rule.Load_calculations=true;
      Quadrature->Rule.Debug=Debug;
      }

    HeterogenousTM(std::string bpn): Temperature(), Periodic_boundary_condition(false), Debug(false), 
                                     Largest_N_in_dataset(),
                                     Zy(), Fy(), Mu_step(0.001),
                                     Y_recalculate(true), Always_recalculate_matrices(false),
                                     pSI(), SI_local(false)
      {
      Quadrature = new quadrature_type();
      Quadrature->Rule.Save_calculations=true;
      Quadrature->Rule.Load_calculations=true;
      Quadrature->Rule.Debug=Debug;
      expand_on_base(bpn);//base used for expanding matrices
      }
      
    inline void debug(bool d) 
      {
      Debug=d;
      Quadrature->Rule.Debug=d;
      }
      
    inline void matrix_base_directory(std::string mbase)
      {
      MatrixDir=mbase;
      Quadrature->Rule.Base_directory=mbase;
      CERR_DEBUG(DHTM_NBDFM) << "New base directory for matrices [" << mbase << "]"  << std::endl;
      }
      
    inline void verbose(bool v) {Verbose=v;}



    /// \brief Adds a new TransferMatrix to the map.
    typename tm_map_type::iterator new_transfer_matrix(const BasePairNeighbours<>& bpn) ///< The base pair neighbours name
      {
      std::string bpname=(std::string)bpn;
      CERR_DEBUG(DHTM_ANMFNN) << " Adding new matrix for NN=" << bpname << std::endl;
      typename tm_map_type::iterator pos=TM_map.find(bpname);
      if (pos == TM_map.end())
        {
        CERR_DEBUG(DHTM_NMYFNN) << " No matrix yet for NN=" << bpname << std::endl;
        std::pair<typename tm_map_type::iterator,bool> newins;
        newins=TM_map.insert(typename tm_map_type::value_type(bpname,GenericTransferMatrix(bpn)));
        if (newins.second) 
          {
          pos=newins.first;
          CERR_DEBUG(DHTM_NMCFNN) << " New matrix created for NN=" << bpname << std::endl;
          delete pos->second.TM->quadrature;
          pos->second.TM->quadrature=Quadrature;
          Range_map[bpn]=Range<double>();
          //looping over A matrices seems not to work yet
          //Range_map[a_matrix(bpn)]=Range<double>();
          return pos;
          }
        else return TM_map.end();
        }
      return pos;
      }

    /// \brief Returns a TransferMatrix identified by the base pair neighbours name.
    transfer_matrix_type* tmatrix(std::string bpn) ///< The base pair neighbours name
      {return TM_map.find(bpn)->second.TM;}

    /// \brief Defines the base on which the expansion takes place.
    void expand_on_base(const BasePairNeighbours<>& bpn)
      {
      BaseName=(std::string)bpn;
      CERR_DEBUG(DHTM_EXPBASE) << " Expanding on base=" << bpn << std::endl;
      Base=TM_map.find((std::string)bpn);
      if (Base == TM_map.end()) 
        {
        CERR_DEBUG(DHTM_EXPBASE) << " Transfer Matrix nos found for " << bpn << " calculating" << std::endl;
        Base=new_transfer_matrix(bpn);
        }
      }

    /// \brief Adds a new basepair .
    void add_nearest_neighbour_pair(const BasePairNeighbours<>& bpn, const tm_matrix_flags& tm_calculation)
      {
      typename tm_map_type::iterator itm;
      itm=TM_map.find((std::string)bpn);

      if (itm == TM_map.end()) 
        {
        itm=new_transfer_matrix(bpn);
        itm->second.Matrix_calculation=tm_calculation;
        itm->second.TM->Hamiltonian_ID=Base->second.TM->Hamiltonian_ID;
        if (tm_calculation == A_only) itm->second.TM->get_basepair_parameters(parameter_map);
        else  itm->second.TM->get_hamiltonian_parameters(parameter_map);
        itm->second.TM->eigenvalue_cutoff(Base->second.TM->Eigenvalue_cutoff);
        itm->second.TM->limits(Base->second.TM->Lower_limit,Base->second.TM->Upper_limit,Base->second.TM->Integration_steps);
        itm->second.TM->temperature(Temperature);
        }
      }

    void select_hamiltonian(std::string hamiltonian_name)
      {
      typename tm_map_type::iterator it;
      for(it=TM_map.begin(); it != TM_map.end(); ++it)
        it->second.TM->select_hamiltonian(hamiltonian_name);
      }

    void print_TM_names(std::ostream& out)
      {
      typename tm_map_type::const_iterator it;
      for(it=TM_map.begin(); it != TM_map.end(); ++it)
        out << it->first << " " << it->second << std::endl;
      out << std::endl;
      }
      
    void reset_all_matrix_calculation(void)
      {
      typename tm_map_type::iterator it;
      for(it=TM_map.begin(); it != TM_map.end(); ++it) it->second.reset_matrix_calculation();
      }

    void all_parameters_used(void)
      {
      typename tm_map_type::iterator it;
      for(it=TM_map.begin(); it != TM_map.end(); ++it) it->second.TM->Parameters_changed=false;
      }
      
    inline std::string hamiltonian(void)
      {
      return Base->second.TM->hamiltonian();
      }

    /// \brief Reads one parameter file and assigns to all maps
    inline bool get_parameters_from_file(std::string par_file)
      {
      parameter_map.read_from(par_file);
      ParameterIdName=parameter_map.Identify;
      bool modified = set_TM_parameters();
      return modified;
      }


    /// \brief Reads a deque of parameter files and assigns to all maps
    /// All parameters are places into parameter_map. The parameter
    /// identification is from the last files
    inline bool get_parameters_from_list_of_files(string_deque_type &par_file_deque)
      {
      parameter_map.read_from_list_of_files(par_file_deque);
      ParameterIdName=parameter_map.Identify;
      bool modified = set_TM_parameters();
      return modified;
      }

    /// \brief sets the parameters for each TM from pre-read parameter_map
    inline bool set_TM_parameters(void)
      {
      typename tm_map_type::iterator it;
      bool modified=false; 
      for(it=TM_map.begin(); it != TM_map.end(); ++it)
        {
        bool this_modified = it->second.get_tm_parameters(parameter_map);
        modified= modified || this_modified;
        }
      return modified;
      }

    void limits(value_type l, ///< Lower integration limit, usually negative 
                value_type u, ///< Upper integration limit
                size_t s)     ///< Number of integration points
      {
      typename tm_map_type::iterator it;
      for(it=TM_map.begin(); it != TM_map.end(); ++it)
        it->second.TM->limits(l,u,s);
      }

    void clear_all_matrices(void)
      {
      reset_all_matrix_calculation();
      Y_recalculate=true;
//       Eigenvector.clear();
//       Eigenvalue.clear();
      }

    void temperature(double T) 
      {
      if (Temperature != T)
        {
        Temperature=T;
        // Whenever the temperature changes we need new matrices.
        clear_all_matrices();

        // Assign the temperature to all known TransferMatrix classes
        typename tm_map_type::iterator it;
        for(it=TM_map.begin(); it != TM_map.end(); ++it)
          it->second.TM->temperature(T) ;
        }
      }

    value_type temperature(void) {return Temperature;}

    /// \brief Maintained for compatibility only.
    ///
    /// This method does nothing, it is kept for compatibility with
    /// certain Action classes.
    void calculate_eigenvalue_only(bool ceo) {}

    void eigenvalue_cutoff(size_t c) 
      {
      typename tm_map_type::iterator it;
      for(it=TM_map.begin(); it != TM_map.end(); ++it)
        it->second.TM->eigenvalue_cutoff(c) ;
      }

    size_t eigenvalue_cutoff(void) 
      {
      size_t N=Base->second.TM->Integration_steps;
      if (Base->second.TM->Eigenvalue_cutoff > (size_t)0) 
        N=Base->second.TM->Eigenvalue_cutoff;
      return N;
      }


    void print_calculation(std::ostream &out)
      {
      for(auto it=TM_map.begin(); it != TM_map.end(); ++it)
        out << *(it->second.TM);
      }

    std::string matrix_filename(std::string mat_name, bp_id_type bp_name=bp_id_type())
      {
      std::stringstream filename;
      filename << MatrixDir << "/" << ParameterIdName
               << "-" << Base->first
               << "-" << Temperature << "-" << mat_name;
      if (bp_name != bp_id_type()) filename << "." << bp_name;
      return filename.str();
      }

    /// \brief Stores matrices into the map and saves into a directory.
    ///
    /// \attention If you don't want to store the matrix in a file
    /// leave MatrixDir empty.
    void save_matrix_to_file(const matrix_type& mat,
                     std::string mat_name, transfer_matrix_type *TM, bp_id_type bp_name=bp_id_type())
      {
      if (!MatrixDir.empty())
        {
        boost::filesystem::create_directory(MatrixDir);
        std::ofstream outfile;
        std::string filename;
        filename=matrix_filename(mat_name,bp_name);
        outfile.open(filename.c_str());
        outfile << *TM << std::endl;
        outfile << mat;
        outfile.close();
        Quadrature->Rule.Base_directory=MatrixDir;
        }
      }
      
    void evaluate_matrices_to_realculate(void)
      {
      if (Base->second.TM->Parameters_changed || Always_recalculate_matrices)
        {
        Base->second.TM->Recalculate_diagonalization=true;
        Y_recalculate=true;
        }
        
      typename tm_map_type::iterator it;
      for(it=TM_map.begin(); it != TM_map.end(); ++it) 
        {
        if (Base->second.TM->Recalculate_diagonalization || it->second.TM->Parameters_changed || Always_recalculate_matrices)
          {
          switch(it->second.Matrix_calculation)
            {
            case A_and_C: it->second.A_matrix_calculated=it->second.ADelta_matrix_calculated=false;
            case C_only : it->second.C_matrix_calculated=it->second.Delta_matrix_calculated=false;
                          break;
            case A_only : it->second.A_matrix_calculated=it->second.ADelta_matrix_calculated=false;
            }
          }
        CERR_DDEBUG(DHTM_PC) << it->first << " Parameters_change=" << it->second.TM->Parameters_changed << " " << it->second << std::endl;
        }
      CERR_DDEBUG_NH(DHTM_PC) << std::endl;
      }
      

    //old name void retrieve_matrices(void)
    void retrieve_or_calculate_matrices(void)
      {
      size_t N=eigenvalue_cutoff();
      matrix_type tmp(N,N);
      tmp.lock_size();
      std::string filename;
      
      evaluate_matrices_to_realculate();
        
      retrieve_eigensystem();
 
      if (Y.size() == (size_t)0 || Y_recalculate)
        {
        filename=matrix_filename("Y");
        if (boost::filesystem::exists(filename)) 
          {
          CERR_DDEBUG(DHTM_READFN) << "Reading " << filename << std::endl;
          std::ifstream in(filename.c_str());
          Y.resize(N,N); Y.lock_size();
          in >> Y;
          Y_recalculate=false;
          }
        else
          {
          if (Verbose) COUT_INFO(INFOFNFC) << filename << " not found, calculating ...";
          calculate_matrix_Y();
          if (Verbose) COUT_INFO(INFOFNFC) << std::endl;
          }
        }

      typename tm_map_type::iterator it;
      for(it=TM_map.begin(); it != TM_map.end(); ++it)
        {
        if (it->second.Use_transpose_matrix) continue;
        CERR_DDEBUG(DHTM_RETCALM) << "TM_Map item " << it->first << std::endl;
        if (it->second.Matrix_calculation != A_only)
          {
          filename=matrix_filename("C",it->first);
          if (boost::filesystem::exists(filename)) 
            {
            CERR_DDEBUG(DHTM_RETCALM) << "Reading " << filename << std::endl;
            std::ifstream in(filename.c_str());
            in >> tmp;
            C_matrix_map[it->first]=tmp;
            it->second.C_matrix_calculated=true;
            }
          else 
            {
            CERR_DDEBUG(DHTM_RETCALM) << filename << " not found, calculating ..."  << std::endl;
            calculate_matrix_C(it);
            CERR_DDEBUG(DHTM_RETCALM) << " calculation done" << std::endl;
            }
          }
        if (it->second.Matrix_calculation != C_only)
          {
          filename=matrix_filename("A",it->first);
          if (boost::filesystem::exists(filename)) 
            {
            CERR_DDEBUG(DHTM_RETCALM) << "Reading " << filename << std::endl;
            std::ifstream in(filename.c_str());
            in >> tmp;
            A_matrix_map[it->first]=tmp;
            it->second.A_matrix_calculated=true;
            }
          else
            {
            CERR_DDEBUG(DHTM_RETCALM) << filename << " not found, calculating ..." << std::endl;
            calculate_matrix_A(it);
            CERR_DDEBUG(DHTM_RETCALM) << "calculation finished" << std::endl;
            }
          }
        }
      
      for(it=TM_map.begin(); it != TM_map.end(); ++it)
        {
        if (it->second.Use_transpose_matrix)
          {
          typename tm_map_type::iterator from=TM_map.find(it->second.Bpn_from);
          switch(it->second.Matrix_calculation)
            {
            case A_and_C: tmp=transpose(A_matrix_map[from->first]);
                          A_matrix_map[it->first]=tmp;
                          it->second.A_matrix_calculated=true;
            case C_only : tmp=transpose(C_matrix_map[from->first]);
                          C_matrix_map[it->first]=tmp;
                          it->second.C_matrix_calculated=true;
                          break;
            case A_only : tmp=transpose(A_matrix_map[from->first]);
                          A_matrix_map[it->first]=tmp;
                          it->second.A_matrix_calculated=true;
                          break;
            }
          }
        }

      generate_delta_matrices();
      all_parameters_used();
      }

    void calculate_eigensystem(void)
      {
      if ( (Eigenvector.size() == (size_t)0) || Base->second.TM->Recalculate_diagonalization)
        {
        CERR_DDEBUG(DHTM_RECDIAG) << " diagonalization of " << Base->first << " Recalculate_diagonalization=" << Base->second.TM->Recalculate_diagonalization << " Parameters_changed=" << Base->second.TM->Parameters_changed << std::endl;

        Base->second.TM->Recalculate_diagonalization=true;
        Base->second.TM->calculate_eigenvalue_only(false);
        Base->second.TM->diagonalize();
        Eigenvector=Base->second.TM->eigen_engine.Eigenvector;
        Eigenvalue=Base->second.TM->eigen_engine.Eigenvalue;
        save_matrix_to_file(Eigenvector,"eigenvectors",Base->second.TM);
        save_matrix_to_file(Eigenvalue,"eigenvalues",Base->second.TM);
        }
      else
        {
        CERR_DDEBUG(DHTM_SKIPDIA) << " skipping diagonalization of " << Base->first << " Recalculate_diagonalization=" << Base->second.TM->Recalculate_diagonalization << " Parameters_changed=" << Base->second.TM->Parameters_changed << std::endl;
        }
      }

    void retrieve_eigensystem(void)
      {
      std::string filename=matrix_filename("eigenvectors");
      if (boost::filesystem::exists(filename)) 
        {
        CERR_DDEBUG(DHTM_RETEIG) << "Reading " << filename  << std::endl;
        std::ifstream in(filename.c_str());
        in >> Eigenvector;
        in.close();
        filename=matrix_filename("eigenvalues");
        CERR_DDEBUG(DHTM_RETEIG) << "Reading " << filename  << std::endl;
        in.open(filename.c_str());
        in >> Eigenvalue;
        in.close();
        }
      else
        {
        CERR_DDEBUG(DHTM_RETEIG) << filename << " not found, calculating ..." << std::endl;
        calculate_eigensystem();
        CERR_DDEBUG(DHTM_RETEIG) << " calculation done" << std::endl;
        }
      }

    void calculate_matrix_AC(typename tm_map_type::iterator it)
      {
      if (it->second.A_matrix_calculated and it->second.C_matrix_calculated) return;

      size_t N=eigenvalue_cutoff();
      matrix_type C(N,N), A(N,N);
      if (it == Base) 
        {
        C.diagonal(Eigenvalue.submatrix(0,0,1,N));
        A=it->second.TM->A_matrix_calculation(Eigenvector);
        }
      else 
        it->second.TM->AC_matrix_calculation(Eigenvector,A,C);
      
      save_matrix_to_file(C,"C",it->second.TM,it->first);
      save_matrix_to_file(A,"A",it->second.TM,it->first);
      
      C_matrix_map[it->first]=C;
      it->second.C_matrix_calculated=true;

      A_matrix_map[it->first]=A;
      it->second.A_matrix_calculated=true;
      }

    void calculate_matrix_C(typename tm_map_type::iterator it)
      {
      if (it->second.C_matrix_calculated) return;
      size_t N=eigenvalue_cutoff();
      matrix_type C(N,N);
      if (it == Base) C.diagonal(Eigenvalue.submatrix(0,0,1,N));
      else C=it->second.TM->C_matrix_calculation(Eigenvector);
      save_matrix_to_file(C,"C",it->second.TM,it->first);
      
      C_matrix_map[it->first]=C;
      it->second.C_matrix_calculated=true;
      }


//     void calculate_all_matrix_C(std::string bp)
//       {
//       typename tm_map_type::iterator it=TM_map.find(bp);
//       if (it != TM_map.end()) calculate_matrix_C(it);
//       }


    void calculate_matrix_A(typename tm_map_type::iterator it)
      {
      if (it->second.A_matrix_calculated) return;
      size_t N=eigenvalue_cutoff();
      matrix_type A(N,N);
      A=it->second.TM->A_matrix_calculation(Eigenvector);
      save_matrix_to_file(A,"A",it->second.TM,it->first);
      A_matrix_map[it->first]=A;
      it->second.A_matrix_calculated=true;
      }


    void calculate_matrix_Y()
      {
      if (Y.size() == (size_t)0 || Y_recalculate)
        {
        Y=Base->second.TM->Y_matrix_calculation(Eigenvector);
        save_matrix_to_file(Y,"Y",Base->second.TM);
        Y_recalculate=false;
        }
      }

    //only used in class ActionZyFy
    //old name: diagonalize
    void calculate_eigensystem_and_matrices(void)
      {
      if (Base->second.TM->Recalculate_diagonalization)
        {
        calculate_eigensystem();
        switch(TM_map[Base->first].Matrix_calculation)
          {
          case A_only : calculate_matrix_A(Base); break;
          case C_only : calculate_matrix_C(Base); break;
          case A_and_C: calculate_matrix_AC(Base);
          }
        calculate_matrix_Y();
     
        typename tm_map_type::iterator it;
        for(it=TM_map.begin(); it != TM_map.end(); ++it)
          {
          if (it != Base)
            {
            switch(it->second.Matrix_calculation)
              {
              case A_only : calculate_matrix_A(it); break;
              case C_only : calculate_matrix_C(it); break;
              case A_and_C: calculate_matrix_AC(it);
              }
            }
          }
        }
      generate_delta_matrices();
      }

    void generate_delta_matrices(void)
      {
      Lambda_matrix=C_matrix_map[BaseName];
      Lambda_matrix.Shape=matrix_type::Diagonal;//This matrix is diagonal by definition
      
      //Here we precaculate Lambda^n which are used n expand_partition_function
      Lambda_matrix_pow.resize(Largest_N_in_dataset+1);
      Lambda_matrix_pow[1]=Lambda_matrix;
      Lambda_matrix_pow[1].Shape=matrix_type::Diagonal;
      for(int n=2; n < (Largest_N_in_dataset+1); n++) 
        {
        Lambda_matrix_pow[n] = Lambda_matrix_pow[n-1]*Lambda_matrix;
        Lambda_matrix_pow[n].Shape=matrix_type::Diagonal;
        }
        
      
      typename tm_map_type::iterator it;
      matrix_type tmp(Lambda_matrix.Rows,Lambda_matrix.Columns);
      for(it=TM_map.begin(); it != TM_map.end(); ++it)
        {
      
        if ( (it->second.Matrix_calculation != A_only) && (!it->second.Delta_matrix_calculated) )
          {
          if (it != Base)
            tmp=C_matrix_map[it->first]-Lambda_matrix;
          else tmp.zero();
          Delta_matrix_map[it->first]=tmp;
          it->second.Delta_matrix_calculated=true;
          }

        if ( (it->second.Matrix_calculation != C_only) && (!it->second.ADelta_matrix_calculated) )
          {
          ADelta_matrix_map[it->first]=A_matrix_map[it->first]-Lambda_matrix;
          it->second.ADelta_matrix_calculated=true;
          }        

        }
      }


    /// \brief Sets the sequence and analyses it.
    ///
    /// By calling this method the class determines which matrices need
    /// to be generated.
    /// This method is used for new sequences passed as strings.
    void sequence(const std::string& seq, const std::string& comp=std::string())
      {
      if (pSI == NULL) 
        {
        pSI=new sequence_info_type;
        SI_local=true;
        }
      pSI->insert_sequence(seq,comp);
      evaluate_sequence_information();
      }
      
    /// \brief Sets the sequence and analyses it.
    ///
    /// By calling this method the class determines which matrices need
    /// to be generated.
    /// This method is used sequences contained in SequenceDataset passed as pointers
    void sequence(sequence_info_type* seqinfo)
      {
      if (pSI != NULL && SI_local) //if a local pSI was generated we delete it first
        {
        delete pSI;
        SI_local=false;
        }
      pSI=seqinfo;
      evaluate_sequence_information();
      }
      
    /// Checks of there are matrices generated for the NN pairs in the sequence
    void evaluate_sequence_information(void)
      {

      typename neighbours_ref_type::const_iterator it;
      for(it=pSI->NPNeighbours_set.begin(); it != pSI->NPNeighbours_set.end(); ++it)
        {
        //Here we resize the Gamma matrix at the sequence evaluation stage so that it needs not to be
        //enlarged during the calculation phase
        size_t N=pSI->Exact_neighbours.size();
        if ( Largest_N_in_dataset < N) Largest_N_in_dataset=N;

        BasePairNeighbours<> Bpn=it->first;
        //Context_dependence now need to be switched off
        Bpn.first.Symmetry_action=BasePairNeighbours<>::internal_type::simplify_symmetry;
        Bpn.second.Symmetry_action=BasePairNeighbours<>::internal_type::simplify_symmetry;
        CERR_DDEBUG(DHTM_CREAMAP) << " Verifying " << (std::string)(it->first) << std::endl;
      
        if (TM_map.find(it->first) == TM_map.end())
          {
          add_nearest_neighbour_pair(Bpn,C_only);
          CERR_DDEBUG(DHTM_CREAMAP) << " Created " << TM_map[it->first] << std::endl; 
          }
        else
          {
          if (TM_map[it->first].Matrix_calculation == A_only)
            {
            CERR_DDEBUG(DHTM_CREAMAP) << " TM matrix for NN=" << (std::string)(it->first) << " is A_only, now becoming A_and_C " << std::endl;

            TM_map[it->first].Matrix_calculation=A_and_C;
            TM_map.find(Bpn)->second.TM->get_hamiltonian_parameters(parameter_map);
            CERR_DDEBUG(DHTM_CREAMAP) << " Changed to A_and_C " << TM_map[it->first] << std::endl; 
            }
          }
          
        BasePairNeighbours<> swp=stack_axis_swap(Bpn);
        CERR_DDEBUG(DHTM_CREAMAP) << " Testing SWP " << swp << " from " << (std::string)(it->first) << std::endl;
        if ( (std::string)Bpn != (std::string)swp )//always use (std::string)
          if ( pSI->Exact_neighbours_set.find(swp) != pSI->Exact_neighbours_set.end() )
           {
           CERR_DDEBUG(DHTM_CREAMAP) << " SWP " << swp << " from " << (std::string)(it->first) << std::endl;
           if (TM_map.find(swp) == TM_map.end())
             {
             TM_map[swp].use_transpose_matrix_from(swp,TM_map[it->first]);
             TM_map[swp].Matrix_calculation=C_only;
             CERR_DDEBUG(DHTM_CREAMAP) << " SWP created " << TM_map[swp] << std::endl; 
             }
           else
             {
             if (TM_map[swp].Matrix_calculation == A_only) 
               {
               TM_map[it->first].Matrix_calculation=TM_map[swp].Matrix_calculation=A_and_C;
               CERR_DDEBUG(DHTM_CREAMAP) << " Changed to A_and_C " << TM_map[it->first] << " " << TM_map[swp] << std::endl; 
               }
             }
           }         
        }
        
      if (TM_map.find(pSI->Periodicity_NN) == TM_map.end())
        {
        BasePairNeighbours<> Bpn=pSI->Periodicity_NN;
        Bpn.first.Symmetry_action=BasePairNeighbours<>::internal_type::simplify_symmetry;
        Bpn.second.Symmetry_action=BasePairNeighbours<>::internal_type::simplify_symmetry;
        if (pSI->NN_periodicity_only) add_nearest_neighbour_pair(Bpn,A_only);
        else                          add_nearest_neighbour_pair(Bpn,A_and_C);
        CERR_DDEBUG(DHTM_CREAMAP) << " Created " << TM_map[pSI->Periodicity_NN] << std::endl; 
        }
      else
        {
        if (TM_map[pSI->Periodicity_NN].Matrix_calculation == C_only) 
          {
          TM_map[pSI->Periodicity_NN].Matrix_calculation=A_and_C;
          CERR_DDEBUG(DHTM_CREAMAP) << " Changed to A_and_C " << TM_map[pSI->Periodicity_NN] << std::endl; 
          }
        }
        
       if ((std::string)pSI->ExactPeriodicity_NN != (std::string)pSI->Periodicity_NN)//always use (std::string)
         {
         if (TM_map.find(pSI->ExactPeriodicity_NN) == TM_map.end())
           {
           TM_map[pSI->ExactPeriodicity_NN].use_transpose_matrix_from(pSI->ExactPeriodicity_NN,TM_map[pSI->Periodicity_NN]);
           if (pSI->NN_periodicity_only) TM_map[pSI->ExactPeriodicity_NN].Matrix_calculation=A_only;
           else                          TM_map[pSI->ExactPeriodicity_NN].Matrix_calculation=A_and_C;
           CERR_DDEBUG(DHTM_CREAMAP) << " ExactPeriodicity_NN created " << TM_map[pSI->ExactPeriodicity_NN] << std::endl; 
           }
         else
          {
          if (TM_map[pSI->ExactPeriodicity_NN].Matrix_calculation == C_only) 
            {
            TM_map[pSI->Periodicity_NN].Matrix_calculation=TM_map[pSI->ExactPeriodicity_NN].Matrix_calculation=A_and_C;
            CERR_DDEBUG(DHTM_CREAMAP) << " ExactPeriodicity_NN Changed to A_and_C " << pSI->ExactPeriodicity_NN << TM_map[pSI->ExactPeriodicity_NN] << std::endl; 
            }
          }
         }

        
      }
      
    inline void get_matrix_eq(matrix_map_type &mp, matrix_type &mat, std::string nm)
      {
      typename matrix_map_type::iterator mpit=mp.find(nm);
      if (mpit == mp.end())
        {
        pSI->print(std::cerr,true);
        print_TM_names(std::cerr);
        print_matrix_list(std::cout,mp);
        CERR_ERROR(ERRNMF) << "No matrix for [" << nm << "]" << std::endl;
        CERR_TERM
        }
      else
        mat=mpit->second;
      }
      
    inline void get_matrix_eq_p(matrix_map_type &mp, typename matrix_map_type::iterator &mpit, std::string nm)
      {
      mpit=mp.find(nm);
      if (mpit == mp.end())
        {
        pSI->print(std::cerr,true);
        print_TM_names(std::cerr);
        print_matrix_list(std::cout,mp);
        CERR_ERROR(ERRNMF) << "No matrix for [" << nm << "]" << std::endl; 
        CERR_TERM   
        }
      }
      
    // Z_y part of Z, presumes matrices are calculated already
    value_type partition_function(void)
      {
      size_t Cut_N=eigenvalue_cutoff();
      size_t N=pSI->Exact_neighbours.size();
      matrix_type Z(Cut_N,Cut_N),C(Cut_N,Cut_N);
      Z.diagonal(1.0);
      NeighbourSequence<>::const_iterator it;
      size_t i;
      for (it=pSI->Exact_neighbours.begin(), i=0; it != pSI->Exact_neighbours.end(); ++it, ++i)
        {
        if (!Periodic_boundary_condition && (i==N-1)) 
          get_matrix_eq(A_matrix_map,C,*it);
        else
          get_matrix_eq(C_matrix_map,C,*it);
        Z=Z*C;
        }

      Zy = Z.trace();
      helmoltz_energy(); //placed into Fy
      return Zy;
      }
      
    // Fy, presumes Zy was calculated already, otherwise will fail because of log(0)
    value_type helmoltz_energy(void)
      {
      value_type kb=GSL_CONST_MKSA_BOLTZMANN/GSL_CONST_MKSA_ELECTRON_VOLT;
      Fy = -kb*Temperature*log(Zy);
      return Fy;
      }

    //This implements the calculations of zhang97b Eqs. (60-64)
    std::valarray<value_type> average_y(void)
      {
      size_t Cut_N=eigenvalue_cutoff();
      matrix_type Z(Cut_N,Cut_N), C(Cut_N,Cut_N);
      
      size_t N=pSI->Exact_neighbours.size();

      std::valarray<value_type> res(N);
      std::vector<matrix_type> F(N+1);
      std::vector<matrix_type> B(N+1);

      NeighbourSequence<>::const_iterator it;
      size_t i;
      Z.diagonal(1.0);
      for (it=pSI->Exact_neighbours.begin(), i=0; it != pSI->Exact_neighbours.end(); ++it, ++i)
        {
        if (!Periodic_boundary_condition && (i==N-1)) 
          get_matrix_eq(A_matrix_map,C,*it);
        else
          get_matrix_eq(C_matrix_map,C,*it); 
        if (i == 0) F[i]=C; else F[i]=F[i-1]*C;
        }
//       if (Debug) std::cerr << std::endl;

      value_type z=F[N-1].trace();
      
      Z.diagonal(1.0);
      B[N]=Z;
      NeighbourSequence<>::reverse_iterator bit;
      for (bit=pSI->Exact_neighbours.rbegin(), i=N-1; bit != pSI->Exact_neighbours.rend(); ++bit, --i)
        {
        if (!Periodic_boundary_condition && (i==(N-1))) 
          get_matrix_eq(A_matrix_map,C,*bit);
        else
          get_matrix_eq(C_matrix_map,C,*bit);
        B[i]=C*B[i+1];
        }

      Z=Y*B[0];//B[0]=C*B[1]*B[2]...*B[N-1]
      res[0]=Z.trace()/z;
      
      for (i=1; i < N; ++i) //here i is the base-pair position
        {
        Z=F[i-1]*Y*B[i];
        res[i]=Z.trace()/z;
        }
      

      return res; //returns Y[i]
      }

    inline std::valarray<value_type> expand_partition_function(void)
      {
      size_t N=pSI->Exact_neighbours.size();
      
      //Gamma is the matrix of the order of Lambda multiplications
      //Gamma[i] = Lambda^i, therefore Gamma[0]=Delta and Gamma[1]=Lambda 
      
      //Not strictly necessary to do both checks, added out of precaution in case expand_partition_function is called without
      //calling first evaluate_sequence_information
      if ( Gamma.size() < (Largest_N_in_dataset+1) ) Gamma.resize(Largest_N_in_dataset+1);
      if ( Gamma.size() < (N+1) ) Gamma.resize(N+1);

      NeighbourSequence<>::const_iterator it=pSI->Exact_neighbours.begin();

      if (it != pSI->Exact_neighbours.end())
        get_matrix_eq(Delta_matrix_map,Gamma[0],(std::string)(*it)); //Here the first delta is directly assigned to Gamma[0]
      else
        {
        CERR_ERROR(ERRVENI) << "Void Exact_neighbours" << std::endl;
        CERR_TERM
        }

      size_t i;
      Gamma[1]=Lambda_matrix;
      for (++it, i=1; it != pSI->Exact_neighbours.end(); ++it, ++i)
        {
        typename matrix_map_type::iterator delta; 
        if (!Periodic_boundary_condition && (i==N-1)) 
          get_matrix_eq_p(ADelta_matrix_map,delta,(std::string)(*it));
        else
          get_matrix_eq_p(Delta_matrix_map,delta,(std::string)(*it));
        
        //Was originally Gamma[i]*Lambda_matrix; 
        //the next line has exactly the same effect but avoids redoing the matrix multiplications every time
        Gamma[i+1]=Lambda_matrix_pow[i+1];    
      
        for (size_t j=i; j > 0; --j) 
          Gamma[j]=Gamma[j]*delta->second+Gamma[j-1]*Lambda_matrix;
        
        Gamma[0]=Gamma[0]*delta->second;
        }

      std::valarray<value_type> res(N+1);
      for (i=0; i < (N+1); ++i)
        res[i]=Gamma[i].trace();
      return res;
      }

    inline value_type melting_index(void)
      {
      std::valarray<value_type> Z_Lambda=expand_partition_function();
      std::valarray<value_type> xi(Z_Lambda.size());

      for(size_t i=0; i < xi.size(); ++i) xi[i]=static_cast<value_type>(i);

      std::valarray<bool> positive=Z_Lambda > value_type();

      std::valarray<value_type> f=Z_Lambda[positive];
      std::valarray<value_type> x=xi[positive];

      value_type s2pi=sqrt(2*M_PI);

      value_type  sigma1=1.0/(f.max()*s2pi);
      value_type  sigma2=sqrt(x.max()-x.min())/2.0;

      std::valarray<value_type> xm=x[f == f.max()];
      value_type  imu = xm[0];

      value_type  best_mu=imu, lowest_std=std::numeric_limits<value_type>::max();
      std::valarray<value_type> gs(x.size()), adf(x.size());
      for (value_type mu=imu-1.0; mu <= imu+1.0; mu+=Mu_step)
        {
        gs=exp(-pow(x-mu,2.0)/(2*pow(sigma2,2.0)))/(sigma1*s2pi);
        adf=pow(f-gs,2.0);
        value_type df=adf.sum();
        if ((df < lowest_std))
          {
          lowest_std=df; 
          best_mu=mu;
          }
        }
        
       return best_mu;
       }

   inline void print_current(std::ostream& out)
      {
      typename range_map_type::iterator rm;
      for (rm=Range_map.begin(); rm != Range_map.end(); ++rm)
         out << rm->second.current() << " ";
      out << std::endl;
      }

    inline void print_matrix_head(std::ostream& out)
      {
      typename matrix_map_type::iterator mm;
      for (mm=C_matrix_map.begin(); mm != C_matrix_map.end(); ++mm)
        {
        std::string mat_name=mm->first;
        if (Range_map.find(mm->first) != Range_map.end()) mat_name="["+mat_name+"]";
        out << std::setw(10) << mat_name << " ";
        }
      out << std::endl;
      }

    inline void print_matrix(std::ostream& out)
      {
      typename matrix_map_type::iterator mm;
      for (mm=C_matrix_map.begin(); mm != C_matrix_map.end(); ++mm)
         out << std::setw(10) << mm->second[0] << " ";
      }
      
    inline void print_matrix_list(std::ostream& out, matrix_map_type &mp)
      {
      typename matrix_map_type::const_iterator mm;
      out << "C_matrix_map list: ";
      for (mm=mp.begin(); mm != mp.end(); ++mm)
         out << mm->first << ", ";
      out << std::endl;
      }


    inline value_type approx_melting_index(void)
      {
      value_type omega=value_type(), lambda=Lambda_matrix[0];
      typename neighbours_ref_type::const_iterator it;
      for (it=pSI->Reduced_neighbours_set.begin(); it != pSI->Reduced_neighbours_set.end(); ++it)
        {
        if (it->second > value_type())
          omega+=it->second*lambda/C_matrix_map[it->first][0];
        }
      // For non-periodic BC we remove one C matrix and add the corresponding A matrix
      if (!Periodic_boundary_condition)
        {
        NeighbourSequence<>::reverse_iterator bit=pSI->Reduced_neighbours.rbegin();
        omega-=lambda/C_matrix_map[*bit][0];
        omega+=lambda/A_matrix_map[*bit][0];
        }
      return omega;
      }

    inline value_type approx_melting_index_loop(void)
      {
      (*Loop_matrix_map)[Base->first][0]=1.0;
      value_type omega=value_type();
      typename neighbours_ref_type::const_iterator it;
      for (it=pSI->Reduced_neighbours_set.begin(); it != pSI->Reduced_neighbours_set.end(); ++it)
        {
        if (it->second > value_type())
          omega+=it->second*(*Loop_matrix_map)[(std::string)(it->first)][0];
        }
      return omega;
      }

    inline value_type cg_content(void)
      {
      value_type cg=0;
      Duplex<>::iterator it=pSI->Exact_duplex.begin();
      for (it=pSI->Exact_duplex.begin(); it != pSI->Exact_duplex.end(); ++it)
        {
        if ((std::string)(*it) == "CG" || (std::string)(*it) == "GC")
          cg+=1.0;
        }
      //std::cout << Sequence << " " << cg << std::endl;
      return cg/pSI->Exact_duplex.size();
      }

    };
  };

#endif
