// $Id: Potentials.h 1251 2021-12-15 18:18:13Z ge $
/// \file Potentials.h
/// \brief contains the basic Potentials function class
///
/// $Revision: 1251 $
/// \author Gerald Weber <gweberbh@gmail.com>
#ifndef GBC_POTENTIALS_H
#define GBC_POTENTIALS_H "$Id: Potentials.h 1251 2021-12-15 18:18:13Z ge $"
#include <valarray>
#include <fstream>
#include "ParameterMap.h"
#include "ErrorCodes.h"

namespace gbc
  {
  template<class _Tp=double>
  /// \brief Base class for potentials
  ///
  /// This base class primarily handles the parameters that are used by all Potentials.
  class BasePotential
    {
    public:
      typedef _Tp value_type;
      typedef std::valarray<value_type> vector_type;
      typedef ParameterMap<value_type>  parameter_map_type;

    private:
      std::string Name;           ///< A name for the potential
      std::string Prefix;         ///< The prefix

    public:
      parameter_map_type ParMap;  ///< Map of parameters used by all potentials that inherit this class

      BasePotential(void): Name(), Prefix(), ParMap() {}

      virtual ~BasePotential() {}

      inline std::string name(void) const {return Name;} 

      inline std::string prefix(void) const {return Prefix;} 

     /// \brief Inserts a prefix in front of the name
      inline void prefix(std::string p) {Prefix=p;}

      inline std::string complete_name(void) const 
        {
        if (Prefix != std::string())
          return Prefix+parameter_map_type::Field_separator+Name;
        else
          return Name;
        } 

      //Same as complete_name() but add a specific prefix
      inline std::string complete_name(std::string prf) const 
        {
        if (prf != std::string())
          return prf+parameter_map_type::Field_separator+Name;
        else
          return Name;
        } 

      inline void name(std::string nm) {Name=nm;} 

    // Placeholder function
    inline bool same_par_values(const BasePotential &pot) const 
      {
      return false;
      } 

    /// \brief This is a placeholder. Each potential should define its own.
    inline virtual bool get_potential_parameters(parameter_map_type& pm)
      {
      CERR_IERROR(IERRPNDEF) << " code problem, please contact developer " << std::endl;
      CERR_TERM
      return true;
      }

    /// \brief Retrieves the parameter from ParameterMap and places it into the local map.
    inline bool define_parameter(parameter_map_type& pm, std::string field, value_type& par, bool complain=true)
      {
      std::string parname=complete_name()+"."+field;
      bool modified=pm.get_parameter(parname,par,complain); //Get the parameter and places it value into par
      ParMap[parname]=par; //Now set the local map
      CERR_DEBUG(DPOT_DFPAR) << " parameter defined \"" << parname << "\" in " << complete_name() << std::endl;
      return modified;
      }

    /// \brief Retrieves the parameter from ParameterMap and places it into the local map.
    inline bool define_pfx_parameter(parameter_map_type& pm, std::string pfx, std::string field, 
                                     value_type& par, bool complain)
      {
      std::string parname=complete_name(pfx)+"."+field;
      bool modified=pm.get_parameter(parname,par,complain); //Get the parameter and places it value into par
      ParMap[parname]=par; //Now set the local map
      return modified;
      }

    /// \brief Retrieves the parameter from ParameterMap and places it into the local map.
    /// This is different from define_pfx_parameter as it returns bool of found or not
    inline bool find_pfx_parameter(parameter_map_type& pm, std::string pfx, std::string field, 
                                     value_type& par, bool& modified, bool complain)
      {
      std::string parname=complete_name(pfx)+"."+field;
      bool found=pm.find_parameter(parname,par,modified,complain); //Get the parameter and places it value into par
      ParMap[parname]=par; //Now set the local map
      return found;
      }

    friend std::ostream& operator<<(std::ostream &out, const BasePotential& pot)
      {
      typename parameter_map_type::const_iterator it;
      out << "[" << pot.complete_name() << "]" << std::endl; 
      for(it=pot.ParMap.begin(); it != pot.ParMap.end(); ++it)
          out << it->first << " " << it->second << std::endl;
      return out;
      }

    inline virtual value_type evaluate_xy(value_type x, value_type y=value_type()) const
      {
      std::cerr << "Warning: evaluate_xy() is only a placeholeder in file " 
                << __FILE__ << " at line " << __LINE__ << std::endl;
      return value_type();
      }

    inline vector_type evaluate(const vector_type& x) const
      {
      size_t x_size=x.size();
      vector_type result(x_size);
      for (size_t i=0; i < x_size; i++) result[i]=evaluate_xy(x[i]); 
      return result;
      }
      
    inline vector_type evaluate(const vector_type& x, const value_type& y) const
      {
      size_t x_size=x.size();
      vector_type result(x_size);
      for (size_t i=0; i < x_size; i++) result[i]=evaluate_xy(x[i],y); 
      return result;
      }

    //vector_type can be both a std::vararray or a matrix type like valmatrix
    inline vector_type evaluate(const vector_type& x, const vector_type& y) const
      {
      size_t nx=x.size(), ny=y.size();
      vector_type result(nx*ny);
      for (size_t i=0; i < nx; i++) 
        {
        for (size_t j=0; j < ny; j++) result[i*ny+j]=evaluate_xy(x[i],y[j]); 
        }
      return result;
      }

    inline vector_type exp_evaluate3D(const vector_type& x, const vector_type& y, const value_type & Beta) const
      {
      size_t nx=x.size(), ny=y.size();
      vector_type result(nx*ny);
      for (size_t i=0; i < nx; i++) 
        {
        for (size_t j=0; j < ny; j++) result[i*ny+j]= sqrt(x[i]*y[j]) * exp(-Beta*evaluate_xy(x[i],y[j])); 
        }
      return result;
      }
    };

  template<class _Tp>
  /// \brief Symmetrizes a one dimensional potential
  ///
  /// The potential \f$ V(x) \f$ is transformed
  /// into a symmetric potential depending on two variables \f$ (x,y) \f$,
  /// \f$ V(x,y)=\frac{1}{2}\left[V(x)+V(y)\right] \f$
  class SymmPotential: public BasePotential<typename _Tp::value_type>
    {
    public:
      typedef _Tp potential_type;
      typedef typename potential_type::value_type value_type;
      typedef std::valarray<value_type> vector_type;
      typedef BasePotential<value_type> base_type;
      typedef typename base_type::parameter_map_type parameter_map_type;
      
      potential_type Vx; ///< The \f$x\f$-dependend potential
      potential_type Vy; ///< The \f$y\f$-dependend potential

    SymmPotential(void): Vx(), Vy() {}

    template<class _Arg>
    SymmPotential(const _Arg& pref): Vx(pref), Vy(pref) {}

    template<class _Arg1, class _Arg2>
    SymmPotential(const _Arg1& prefVx,const _Arg2& prefVy): Vx(prefVx), Vy(prefVy) {}

    inline bool same_par_values(const SymmPotential &pot) const 
      {
      return Vx.same_par_values(pot.Vx) and Vy.same_par_values(pot.Vy);
      } 

    // Checks if Vx parameters equals Vy and vice-versa
    inline bool switched_par_values(const SymmPotential &pot) const 
      {
      return Vy.same_par_values(pot.Vx) and Vx.same_par_values(pot.Vy);
      } 

    inline virtual bool get_potential_parameters(parameter_map_type& pm)
      {
      bool modified_vx=Vx.get_potential_parameters(pm);
      bool modified_vy=Vy.get_potential_parameters(pm);
      return modified_vx || modified_vy;
      }

    inline bool parameters(const potential_type& px) 
      {
      bool modified_vx=Vx.parameters(px);
      bool modified_vy=Vy.parameters(px);
      return modified_vx || modified_vy;
      }

    inline bool parameters(const potential_type& px, const potential_type& py) 
      {
      bool modified_vx=Vx.parameters(px);
      bool modified_vy=Vy.parameters(py);
      return modified_vx || modified_vy;
      }
 
    inline bool parameters(const SymmPotential& p) 
      {
      bool modified_vx=Vx.parameters(p.Vx);
      bool modified_vy=Vy.parameters(p.Vy);
      return modified_vx || modified_vy;
      }

    friend std::ostream& operator<<(std::ostream &out, const SymmPotential& sp)
      {
      out << "SymmPotential Vx: " << sp.Vx << " Vy: " << sp.Vy;
      return out;
      }


     /// \brief Evaluates the symmetrized potential 
     ///
     /// \f$V(x,y)=\frac{1}{2}\left[V(x)+V(y)\right] \f$
    inline value_type evaluate_xy(value_type x, value_type y=value_type()) const
      {
      return 0.5*(Vx.evaluate_xy(x)+Vy.evaluate_xy(y));
      }
    };

  template<class _FirstPot,class _SecondPot>
  /// \brief Combines two potentials.
  ///
  /// The two potentials can be individually accessed through the First/Second
  /// objects. 
  class CmbPotential: public BasePotential<typename _FirstPot::value_type>
    {
    public:
      typedef _FirstPot first_potential_type;
      typedef _SecondPot second_potential_type;
      typedef typename first_potential_type::value_type value_type;
      typedef std::valarray<value_type> vector_type;
      typedef BasePotential<value_type> base_type;
      typedef typename base_type::parameter_map_type parameter_map_type;
      
      first_potential_type First;
      second_potential_type Second;

    CmbPotential(void): First(), Second() {}

    template<class _Arg>
    CmbPotential(const _Arg& arg): First(arg), Second(arg)
      {
      };

    template<class _Arg1, class _Arg2>
    CmbPotential(const _Arg1& p1, const _Arg2& p2): First(p1), Second(p2)
      {
      };

    template<class _Arg1, class _Arg2, class _Arg3>
    CmbPotential(const _Arg1& p1, const _Arg2& p2, const _Arg3& p3)
      : First(p1,p2), Second(p3)
      {
      };

    inline bool same_par_values(const CmbPotential &pot) const 
      {
      return First.same_par_values(pot.First) and Second.same_par_values(pot.Second);
      } 

    inline virtual bool get_potential_parameters(parameter_map_type& pm)
      {
      bool modified_f=First.get_potential_parameters(pm);
      bool modified_s=Second.get_potential_parameters(pm);
      return modified_f || modified_s;
      }

    inline bool parameters(const first_potential_type& pf, const second_potential_type& ps) 
      {
      bool modified_f=First.parameters(pf);
      bool modified_s=Second.parameters(ps);
      return modified_f || modified_s;}

    inline bool parameters(const CmbPotential& p) 
      {return parameters(p.First,p.Second);}

    friend std::ostream& operator<<(std::ostream &out, const CmbPotential& cp)
      {
      out << "CmbPotential First: " << cp.First << " Second: " << cp.Second;
      return out;
      }

    /// \brief Evaluates the combined potential by summing each potential.
    inline value_type evaluate_xy(value_type x, value_type y=value_type()) const
      {
      return First.evaluate_xy(x,y)+Second.evaluate_xy(x,y);
      }
    };

  };
#endif
